! ==============================================================================
! Subroutine: BOND (EBOND,FTOT)
! 
! Purpose: Computes the energy, force and stress tensor due to bond
!          stretching in a covalent bond.
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
!          The energy due to bond stretching is:
!
!          E = E0 * (D - DEQ)^2
!
!          where E0 = 1/2 the Spring Constant, and DEQ is the
!          equilibrium bond length between the two atoms.
!
! Arguments:
!
!           EBOND - Energy due to covalent bond stretching.
!           FTOT  - Array of dimension (3,NAT) containing the total
!                   force on each atom.
!           VIR   - Array of dimension (3,3) containing the virial
!                   tensor with elements given by:
!                   S_ab = SUM_i Fia * Rib where Fia = ath
!                   component of the force on atom i.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE BOND (EBOND,FTOT)

        USE SystemParam, ONLY : rat,vir,dubd,lubd,nat,nubd,ipbc,ibd

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(OUT) :: ebond
        DOUBLE PRECISION, INTENT(INOUT) :: ftot(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k

        DOUBLE PRECISION :: x1,x2,x3,s11,s12,s13,s22,s23,s33
        DOUBLE PRECISION :: a1,a2,a3,c,d,deq,spr,arg


        ebond = 0.0d0

        IF ( ibd == 0 .or. nubd == 0 ) RETURN


        DO i=1,nubd

          j = lubd(1,i)
          k = lubd(2,i)

          spr = dubd(1,i)
          deq = dubd(2,i)

          x1 = rat(1,j) - rat(1,k)
          x2 = rat(2,j) - rat(2,k)
          x3 = rat(3,j) - rat(3,k)

          d = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(d)

          arg = d - deq
          c = spr * arg

          ebond = ebond + c * arg

          c = 2.0d0 * c / d

          a1 = c * x1
          a2 = c * x2
          a3 = c * x3

          ftot(1,j) = ftot(1,j) - a1
          ftot(2,j) = ftot(2,j) - a2
          ftot(3,j) = ftot(3,j) - a3

          ftot(1,k) = ftot(1,k) + a1
          ftot(2,k) = ftot(2,k) + a2
          ftot(3,k) = ftot(3,k) + a3

          IF ( ipbc /= 2 ) CYCLE

          s11 = a1 * x1
          s12 = a1 * x2
          s13 = a1 * x3
          s22 = a2 * x2 
          s23 = a2 * x3
          s33 = a3 * x3

          vir(1,1) = vir(1,1) - s11
          vir(2,1) = vir(2,1) - s12
          vir(3,1) = vir(3,1) - s13
          vir(1,2) = vir(1,2) - s12
          vir(2,2) = vir(2,2) - s22
          vir(3,2) = vir(3,2) - s23
          vir(1,3) = vir(1,3) - s13
          vir(2,3) = vir(2,3) - s23
          vir(3,3) = vir(3,3) - s33

        ENDDO

        RETURN

      END SUBROUTINE BOND
