! ==============================================================================
! Subroutine: DIHEDRAL (EDIHD,FTOT)
! 
! Purpose: Computes the energy, force and stress tensor due to a dihedral
!          rotation about a covalent bond.
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
!          The energy due to a dihedral rotation is:
!
!          E = E0 * (1 + Cos[N*PHI - GAMMA])
!
!          where E0 = the Energy Constant, N is the dihedral multiplicity,
!          PHI is the dihedral angle, and GAMMA is the dihedral phase angle.
!
! Arguments:
!
!           EDIHD - Energy due to a dihedral rotation about a covalent
!                   bond.
!           FTOT  - Array of dimension (3,NAT) containing the total
!                   force on each atom.
!           VIR   - Array of dimension (3,3) containing the virial
!                   tensor with elements given by:
!                   S_ab = SUM_i Fia * Rib where Fia = ath
!                   component of the force on atom i.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE DIHEDRAL (EDIHD,FTOT)

        USE SystemParam, ONLY : rat,vir,ddihd,ldihd,nat,ndh,ipbc,idh

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(OUT) :: edihd
        DOUBLE PRECISION, INTENT(INOUT) :: ftot(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m,n

        DOUBLE PRECISION :: x1,x2,x3,y1,y2,y3,z1,z2,z3
        DOUBLE PRECISION :: a1,a2,a3,b1,b2,b3,c1,c2,c3
        DOUBLE PRECISION :: s11,s12,s13,s22,s23,s33,d1,d2,d3
        DOUBLE PRECISION :: arg,asq,bsq,phi,spr,gam,pn,c,d

        DOUBLE PRECISION, PARAMETER :: sml = 1.00d-6
        DOUBLE PRECISION, PARAMETER :: twopi = 6.28318530717958623199d0


        edihd = 0.0d0

        IF ( idh == 0 .or. ndh == 0 ) RETURN


        DO i=1,ndh

          j = ldihd(1,i)
          k = ldihd(2,i)
          m = ldihd(3,i)
          n = ldihd(4,i)

          spr = ddihd(1,i)
          gam = ddihd(2,i)
          pn = ddihd(3,i)

          x1 = rat(1,j) - rat(1,k)
          x2 = rat(2,j) - rat(2,k)
          x3 = rat(3,j) - rat(3,k)

          y1 = rat(1,m) - rat(1,k)
          y2 = rat(2,m) - rat(2,k)
          y3 = rat(3,m) - rat(3,k)

          z1 = rat(1,n) - rat(1,m)
          z2 = rat(2,n) - rat(2,m)
          z3 = rat(3,n) - rat(3,m)

          a1 = x2 * y3 - x3 * y2
          a2 = x3 * y1 - x1 * y3
          a3 = x1 * y2 - x2 * y1

          b1 = z2 * y3 - z3 * y2
          b2 = z3 * y1 - z1 * y3
          b3 = z1 * y2 - z2 * y1

          asq = a1 * a1 + a2 * a2 + a3 * a3
          bsq = b1 * b1 + b2 * b2 + b3 * b3

          d = a1 * b1 + a2 * b2 + a3 * b3
          c = a1 * z1 + a2 * z2 + a3 * z3

          IF ( asq < sml ) THEN
            asq = 0.0d0
          ELSE
            asq = 1.0d0 / asq
          ENDIF

          IF ( bsq < sml ) THEN
            bsq = 0.0d0
          ELSE
            bsq = 1.0d0 / bsq
          ENDIF

          d1 = asq * bsq
          d1 = DSQRT(d1)

          phi = d * d1

          phi = MIN(+1.0d0,phi)
          phi = MAX(-1.0d0,phi)

          phi = DACOS(phi)

          IF ( c > 0.0d0 ) phi = twopi - phi

          arg = pn * phi - gam

          edihd = edihd + spr * ( 1.0d0 + DCOS(arg) )

          c = spr * pn * d1

          d1 = DSIN(phi)

          IF ( DABS(d1) > sml ) THEN
            c = c * DSIN(arg) / d1
          ELSE
            c = c * pn * DCOS(arg)
            c = c * DSIGN(1.0d0,DCOS(phi))
          ENDIF

          d1 = d * asq
          d2 = d * bsq

          c1 = d2 * b1 - a1
          c2 = d2 * b2 - a2
          c3 = d2 * b3 - a3

          b1 = d1 * a1 - b1
          b2 = d1 * a2 - b2
          b3 = d1 * a3 - b3

          a1 = c * ( y2 * b3 - y3 * b2 )
          a2 = c * ( y3 * b1 - y1 * b3 )
          a3 = c * ( y1 * b2 - y2 * b1 )

          d1 = x2 * b3 - x3 * b2
          d2 = x3 * b1 - x1 * b3
          d3 = x1 * b2 - x2 * b1

          b1 = c * ( d1 + z2 * c3 - z3 * c2 )
          b2 = c * ( d2 + z3 * c1 - z1 * c3 )
          b3 = c * ( d3 + z1 * c2 - z2 * c1 )

          d1 = c * ( y2 * c3 - y3 * c2 )
          d2 = c * ( y3 * c1 - y1 * c3 )
          d3 = c * ( y1 * c2 - y2 * c1 )

          ftot(1,j) = ftot(1,j) + a1
          ftot(2,j) = ftot(2,j) + a2
          ftot(3,j) = ftot(3,j) + a3

          ftot(1,k) = ftot(1,k) - a1 + b1
          ftot(2,k) = ftot(2,k) - a2 + b2
          ftot(3,k) = ftot(3,k) - a3 + b3

          ftot(1,m) = ftot(1,m) - d1 - b1
          ftot(2,m) = ftot(2,m) - d2 - b2
          ftot(3,m) = ftot(3,m) - d3 - b3

          ftot(1,n) = ftot(1,n) + d1
          ftot(2,n) = ftot(2,n) + d2
          ftot(3,n) = ftot(3,n) + d3

          IF ( ipbc /= 2 ) CYCLE

          s11 = a1 * x1 - b1 * y1 + d1 * z1
          s12 = a1 * x2 - b1 * y2 + d1 * z2
          s13 = a1 * x3 - b1 * y3 + d1 * z3
          s22 = a2 * x2 - b2 * y2 + d2 * z2
          s23 = a2 * x3 - b2 * y3 + d2 * z3
          s33 = a3 * x3 - b3 * y3 + d3 * z3

          vir(1,1) = vir(1,1) + s11
          vir(2,1) = vir(2,1) + s12
          vir(3,1) = vir(3,1) + s13
          vir(1,2) = vir(1,2) + s12
          vir(2,2) = vir(2,2) + s22
          vir(3,2) = vir(3,2) + s23
          vir(1,3) = vir(1,3) + s13
          vir(2,3) = vir(2,3) + s23
          vir(3,3) = vir(3,3) + s33

        ENDDO

        RETURN

      END SUBROUTINE DIHEDRAL
