! ==============================================================================
! Subroutine: ECAVITY (ECAV,FTOT)
! 
! Purpose: Computes the free energy, and force due to cavitation.
!
! Method:  (1) W.C. Still et al., J. Comp. Chem., 20 217 (1999).
!          (2) J. Phys. Chem., 98 1978-1988 (1994).
!
!          The energy of cavitation is:
!
!          ECAV = SIGMA * SASA + RHO
!
!          where SASA is the Solvent Accessable Surface Area and
!          SIGMA is the free energy cost per unit area for forming
!          a cavity in the solvent and RHO is a free energy offset.
!
! Arguments:
!
!           ECAV - Energy due to Coulomb interactions.
!           FTOT - Array of dimension (3,NAT) containing the force
!                  force on each atom.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE ECAVITY (ECAV,FTOT)

        USE SystemParam, ONLY : rat,dlv,grot,gbparm,indgb,itype,lnpair,&
                              & nnpair,nat,igbsa,isim,isym

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(OUT) :: ecav
        DOUBLE PRECISION, INTENT(INOUT) :: ftot(3,nat)

        !=== VARIABLES ===!

        INTEGER, DIMENSION(:,:,:), ALLOCATABLE :: list
        DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: r1,r2,r3

        INTEGER :: i,j,k,m,n,m1,m2,m3,i1,i2,nmax
        INTEGER :: itp,jtp,ktp,istep,nn(nat)

        DOUBLE PRECISION :: x1,x2,x3,y1,y2,y3,a1,a2,a3,aij
        DOUBLE PRECISION :: xsq,ysq,a,b,d,ai,aj,ak,s2,s3,s4
        DOUBLE PRECISION :: sasa,p1,p2,p3,p4

        DOUBLE PRECISION, PARAMETER :: pi = 3.14159265358979323846d0
        DOUBLE PRECISION, PARAMETER :: sigma = 2.1682050391d-4
        DOUBLE PRECISION, PARAMETER :: rho = 3.7293126672d-2


        ecav = 0.0d0
        sasa = 0.0d0

        IF ( igbsa == 0 .or. isim == 0 ) RETURN


        !=== Determine SASA Neighbors ===!

        istep = 1

 1      nn = 0

        DO i=1,nnpair

          j = lnpair(1,i)
          k = lnpair(2,i)
          m = lnpair(3,i)

          jtp = indgb(j)
          ktp = indgb(k)

          aj = gbparm(1,jtp)
          ak = gbparm(1,ktp)

          IF ( itype(j) == 1 ) CYCLE
          IF ( itype(k) == 1 ) CYCLE

          IF ( isym == 0 .or. m == 0 ) THEN

            m2 = MODULO(m,1000000)
            m3 = MODULO(m2,1000)
            m1 = m / 1000000
            m2 = m2 / 1000

            IF ( m1 > 100 ) m1 = 100 - m1
            IF ( m2 > 100 ) m2 = 100 - m2
            IF ( m3 > 100 ) m3 = 100 - m3

            a1 = DBLE(m1) * dlv(1,1) + DBLE(m2) * dlv(1,2) &
             & + DBLE(m3) * dlv(1,3)
            a2 = DBLE(m1) * dlv(2,1) + DBLE(m2) * dlv(2,2) &
             & + DBLE(m3) * dlv(2,3)
            a3 = DBLE(m1) * dlv(3,1) + DBLE(m2) * dlv(3,2) &
             & + DBLE(m3) * dlv(3,3)

            x1 = rat(1,j) - rat(1,k) - a1
            x2 = rat(2,j) - rat(2,k) - a2
            x3 = rat(3,j) - rat(3,k) - a3

          ELSE

            x1 = rat(1,j) - grot(1,1,m) * rat(1,k) &
             & - grot(1,2,m) * rat(2,k) &
             & - grot(1,3,m) * rat(3,k)

            x2 = rat(2,j) - grot(2,1,m) * rat(1,k) &
             & - grot(2,2,m) * rat(2,k) &
             & - grot(2,3,m) * rat(3,k)

            x3 = rat(3,j) - grot(3,1,m) * rat(1,k) &
             & - grot(3,2,m) * rat(2,k) &
             & - grot(3,3,m) * rat(3,k)

          ENDIF

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)

          a = aj + ak

          IF ( d < a ) THEN

            nn(j) = nn(j) + 1

            IF ( isym == 0 .or. m == 0 ) THEN
            nn(k) = nn(k) + 1
            ENDIF

            IF ( istep == 2 ) THEN

              list(1,nn(j),j) = k
              list(2,nn(j),j) = m

              IF ( isym == 0 .or. m == 0 ) THEN

                m1 = -m1
                m2 = -m2
                m3 = -m3

                IF ( m1 < 0 ) m1 = 100 + IABS(m1)
                IF ( m2 < 0 ) m2 = 100 + IABS(m2)
                IF ( m3 < 0 ) m3 = 100 + IABS(m3)

                m = m1 * 1000000 + m2 * 1000 + m3

                list(1,nn(k),k) = j
                list(2,nn(k),k) = m

              ENDIF

            ENDIF

          ENDIF

        ENDDO

        IF ( istep == 1 ) THEN

          nmax = 0

          DO i=1,nat
          IF ( nn(i) > nmax ) nmax = nn(i)
          ENDDO

          nmax = MAX(1,nmax)

          ALLOCATE (list(2,nmax,nat))
          ALLOCATE (r1(nmax),r2(nmax),r3(nmax))

          istep = 2

          GOTO 1

        ENDIF


        !=== Calculate SASA ===!

        DO i=1,nat

          itp = indgb(i)

          ai = gbparm(1,itp)
          p1 = gbparm(2,itp)
          p2 = gbparm(3,itp)
          p3 = gbparm(4,itp)
          p4 = gbparm(5,itp)

          !=== First Term ===!

          a = 2.0d0 * ai
          a = pi * ai * ai

          sasa = sasa + p1 * a

          !=== Get Neighbor Positions ===!

          n = nn(i)
          IF ( n == 0 ) CYCLE

          DO i1=1,n

            j = list(1,i1,i)
            m = list(2,i1,i)

            IF ( isym == 0 .or. m == 0 ) THEN

              m2 = MODULO(m,1000000)
              m3 = MODULO(m2,1000)
              m1 = m / 1000000
              m2 = m2 / 1000

              IF ( m1 > 100 ) m1 = 100 - m1
              IF ( m2 > 100 ) m2 = 100 - m2
              IF ( m3 > 100 ) m3 = 100 - m3

              a1 = DBLE(m1) * dlv(1,1) + DBLE(m2) * dlv(1,2) &
               & + DBLE(m3) * dlv(1,3)
              a2 = DBLE(m1) * dlv(2,1) + DBLE(m2) * dlv(2,2) &
               & + DBLE(m3) * dlv(2,3)
              a3 = DBLE(m1) * dlv(3,1) + DBLE(m2) * dlv(3,2) &
               & + DBLE(m3) * dlv(3,3)

              r1(i1) = rat(1,j) + a1
              r2(i1) = rat(2,j) + a2
              r3(i1) = rat(3,j) + a3

            ELSE

              r1(i1) = grot(1,1,m) * rat(1,j) &
                   & + grot(1,2,m) * rat(2,j) &
                   & + grot(1,3,m) * rat(3,j)

              r2(i1) = grot(2,1,m) * rat(1,j) &
                   & + grot(2,2,m) * rat(2,j) &
                   & + grot(2,3,m) * rat(3,j)

              r3(i1) = grot(3,1,m) * rat(1,j) &
                   & + grot(3,2,m) * rat(2,j) &
                   & + grot(3,3,m) * rat(3,j)

            ENDIF

          ENDDO


          !=== Second - Fourth Terms ===!

          s2 = 0.0d0
          s3 = 0.0d0
          s4 = 0.0d0

          DO i1=1,n

            j = list(1,i1,i)

            jtp = indgb(j)
            aj = gbparm(1,jtp)

            !=== Second Term ===!

            x1 = rat(1,i) - r1(i1)
            x2 = rat(2,i) - r2(i1)
            x3 = rat(3,i) - r3(i1)

            xsq = x1 * x1 + x2 * x2 + x3 * x3
            d = DSQRT(xsq)

            a = ai * ai - aj * aj
            a = a / xsq

            b = 1.0d0 + a
            b = 2.0d0 * ai - d * b

            aij = pi * ai * b

            s2 = s2 + aij

            b = pi * ai * ( 1.0d0 - a )
            b = b / d

            x1 = b * x1
            x2 = b * x2
            x3 = b * x3

            !=== Third Term ===!

            DO i2=1,n
            IF ( i1 /= i2 ) THEN

              k = list(1,i2,i)

              ktp = indgb(k)
              ak = gbparm(1,ktp)

              y1 = r1(i1) - r1(i2)
              y2 = r2(i1) - r2(i2)
              y3 = r3(i1) - r3(i2)

              ysq = y1 * y1 + y2 * y2 + y3 * y3
              d = DSQRT(ysq)

              a = aj + ak

              IF ( d > a ) CYCLE

              a = aj * aj - ak * ak
              a = a / ysq

              b = 1.0d0 + a
              b = 2.0d0 * aj - d * b

              s3 = s3 + pi * aj * b

              b = pi * aj * ( 1.0d0 - a )
              b = b * ( p3 + aij * p4 ) / d

              y1 = b * y1
              y2 = b * y2
              y3 = b * y3

              ftot(1,j) = ftot(1,j) + y1
              ftot(2,j) = ftot(2,j) + y2
              ftot(3,j) = ftot(3,j) + y3

              ftot(1,k) = ftot(1,k) - y1
              ftot(2,k) = ftot(2,k) - y2
              ftot(3,k) = ftot(3,k) - y3

            ENDIF
            ENDDO

            !=== Fourth Term ===!

            s4 = s4 + aij * s3

            b = p2 + p4 * s3

            x1 = b * x1
            x2 = b * x2
            x3 = b * x3

            ftot(1,i) = ftot(1,i) + x1
            ftot(2,i) = ftot(2,i) + x2
            ftot(3,i) = ftot(3,i) + x3

            ftot(1,j) = ftot(1,j) - x1
            ftot(2,j) = ftot(2,j) - x2
            ftot(3,j) = ftot(3,j) - x3

          ENDDO

          sasa = sasa + p2 * s2 + p3 * s3 + p4 * s4

        ENDDO


        !=== Solvation Energy ===!

        ecav = sigma * sasa + rho

        !=== Deallocate Arrays ===!

        DEALLOCATE (list,r1,r2,r3)

        RETURN

      END SUBROUTINE ECAVITY
