! ==============================================================================
! Subroutine: ENERGY (EBOND,EANGL,EDIHD,ECOUL,EVDW,FTOT)
! 
! Purpose: Computes the TOTAL energy, force and stress tensor due to
!          a classical molecular dynamics force field (e.g. AMBER).
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
! Arguments:
!
!           EBOND - Energy due to covalent bond stretching.
!           EANGL - Energy due to angle bending between two
!                   covalent bonds.
!           EDIHD - Energy due to dihedral rotations about
!                   covalent bonds.
!           ECOUL - Energy due to Coulomb interactions.
!           EVDW  - Energy due to van der Waals interactions.
!           FTOT  - Array of dimension (3,NAT) containing the total
!                   force on each atom.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - BOND, ANGLE, DIHEDRAL, GBORN, ECAVITY, ESTATIC
!               EWALD_DIR, EWALD_REC, EWALD_PM, VDW, EXCLUDED, SCALED
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE ENERGY (EBOND,EANGL,EDIHD,ECOUL,EVDW,FTOT)

        USE SystemParam, ONLY : vir,nat,ipbc,ircp,igb,igbsa

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(OUT) :: ebond,eangl,edihd,ecoul
        DOUBLE PRECISION, INTENT(OUT) :: evdw,ftot(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i
        DOUBLE PRECISION :: edir,erec,ecav,ecex,ecsc,evex,evsc


        ebond = 0.0d0
        eangl = 0.0d0
        edihd = 0.0d0
        ecoul = 0.0d0
        evdw = 0.0d0

        edir = 0.0d0
        erec = 0.0d0
        ecav = 0.0d0
        ecex = 0.0d0
        evex = 0.0d0
        ecsc = 0.0d0
        evsc = 0.0d0

        ftot = 0.0d0
        vir = 0.0d0


        !=== Bond Terms ===!

        CALL BOND (ebond,ftot)

        !=== Angle Terms ===!

        CALL ANGLE (eangl,ftot)

        !=== Dihedral Terms ===!

        CALL DIHEDRAL (edihd,ftot)


        !=== Non-Bonded Terms ===!

        !=== Generalized Born Model ===!

        IF ( igb /= 0 ) THEN

          CALL GBORN (ecoul,evdw,ftot)

          CALL ECAVITY (ecav,ftot)

          ecoul = ecoul + ecav

        ENDIF

        !=== No Periodic Boundary ===!

        IF ( igb == 0 .and. ipbc == 0 ) THEN

          CALL ESTATIC (ecoul,evdw,ftot)

        ENDIF

        !=== Periodic Boundary ===!

        IF ( igb == 0 .and. ipbc /= 0 ) THEN

          CALL EWALD_DIR (edir,ftot)

          IF ( ircp == 1 ) THEN

            CALL EWALD_REC (erec,ftot)

          ELSEIF ( ircp == 2 ) THEN

            CALL EWALD_PM (erec,ftot)

          ENDIF

          ecoul = edir + erec

          CALL VDW (evdw,ftot)

        ENDIF

        !=== Non-Bonded Corrections ===!

        CALL EXCLUDED (ecex,evex,ftot)

        CALL SCALED (ecsc,evsc,ftot)

        ecoul = ecoul + ecex + ecsc

        evdw = evdw + evex + evsc

        RETURN

      END SUBROUTINE ENERGY
