! ==============================================================================
! Subroutine: ESTATIC (ECOUL,EVDW,FTOT)
! 
! Purpose: Computes the energy and force due to electrostatic interactions
!          between two atoms in a system. A cutoff is used to truncate
!          the Coulomb and van der Waals sums.
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
!          The energy due to electrostatic interactions is:
!
!          E = Q'i * Q'j / R + A / R^12 - B / R^6
!
!          where R is the distance between atoms i and j
!          A = EPS * ( REQ )**12 and B = 2 * EPS * ( REQ )**6
!          REQ is given by the sum of the van der Waals radii for
!          each atom and EPS is the well depth.
!
!          NOTE: Q'i is the scaled charge Qi / SQRT(eps)
!
! Arguments:
!
!           ECOUL - Energy due to Coulomb interactions.
!           EVDW  - Energy due to van der Waals interactions.
!           FTOT  - Array of dimension (3,NAT) containing the total
!                   force on each atom.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - SMOOTH
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE ESTATIC (ECOUL,EVDW,FTOT)

        USE SystemParam, ONLY : rat,q,dvdw,dlv,cut,scut,grot,nat,&
                              & lnpair,nnpair,isym,icou,ivdw,ismth

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(OUT) :: ecoul,evdw
        DOUBLE PRECISION, INTENT(INOUT) :: ftot(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m,m1,m2,m3

        DOUBLE PRECISION :: x1,x2,x3,a1,a2,a3,req,eps
        DOUBLE PRECISION :: a,b,c,d,e,s,ds,dds,xsq


        ecoul = 0.0d0
        evdw = 0.0d0

        IF ( nnpair == 0 ) RETURN
        IF ( icou == 0 .and. ivdw == 0 ) RETURN


        DO i=1,nnpair

          a = 0.0d0
          b = 0.0d0
          c = 0.0d0
          s = 1.0d0
          e = 0.0d0

          j = lnpair(1,i)
          k = lnpair(2,i)
          m = lnpair(3,i)

          IF ( isym == 0 .or. m == 0 ) THEN

            m2 = MODULO(m,1000000)
            m3 = MODULO(m2,1000)
            m1 = m / 1000000
            m2 = m2 / 1000

            IF ( m1 > 100 ) m1 = 100 - m1
            IF ( m2 > 100 ) m2 = 100 - m2
            IF ( m3 > 100 ) m3 = 100 - m3

            a1 = DBLE(m1) * dlv(1,1) + DBLE(m2) * dlv(1,2) &
             & + DBLE(m3) * dlv(1,3)
            a2 = DBLE(m1) * dlv(2,1) + DBLE(m2) * dlv(2,2) &
             & + DBLE(m3) * dlv(2,3)
            a3 = DBLE(m1) * dlv(3,1) + DBLE(m2) * dlv(3,2) &
             & + DBLE(m3) * dlv(3,3)

            x1 = rat(1,j) - rat(1,k) - a1
            x2 = rat(2,j) - rat(2,k) - a2
            x3 = rat(3,j) - rat(3,k) - a3

          ELSE

            x1 = rat(1,j) - grot(1,1,m) * rat(1,k) &
             & - grot(1,2,m) * rat(2,k) &
             & - grot(1,3,m) * rat(3,k)

            x2 = rat(2,j) - grot(2,1,m) * rat(1,k) &
             & - grot(2,2,m) * rat(2,k) &
             & - grot(2,3,m) * rat(3,k)

            x3 = rat(3,j) - grot(3,1,m) * rat(1,k) &
             & - grot(3,2,m) * rat(2,k) &
             & - grot(3,3,m) * rat(3,k)

          ENDIF

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)

          IF ( ismth /= 0 .and. d > scut ) THEN

            CALL SMOOTH (d,scut,cut,s,ds,dds)

          ENDIF

          IF ( icou == 1 ) THEN

            c = q(j) * q(k) / d

            ecoul = ecoul + c * s

          ENDIF

          IF ( ivdw == 1 ) THEN

            req = dvdw(1,j) + dvdw(1,k)
            eps = dvdw(2,j) * dvdw(2,k)
            eps = DSQRT(eps)

            a = req / d

            a = a * a
            b = a * a
            a = a * b

            b = eps * a
            a = a * b
            b = 2.0d0 * b

            e = a - b

            evdw = evdw + e * s

          ENDIF

          a = 12.0d0 * a - 6.0d0 * b + c
          a = a / xsq

          IF ( ismth /= 0 .and. d > scut ) THEN

            c = e + c
            c = c / d

            a = a * s - c * ds

          ENDIF

          a1 = a * x1
          a2 = a * x2
          a3 = a * x3

          ftot(1,j) = ftot(1,j) + a1
          ftot(2,j) = ftot(2,j) + a2
          ftot(3,j) = ftot(3,j) + a3

          ftot(1,k) = ftot(1,k) - a1
          ftot(2,k) = ftot(2,k) - a2
          ftot(3,k) = ftot(3,k) - a3

        ENDDO

        RETURN

      END SUBROUTINE ESTATIC
