! ==============================================================================
! Subroutine: EWALD_DIR (EDIR,FTOT)
! 
! Purpose: Computes the electrostatic energy, forces, and stress tensor of
!          a system of charges with periodic boundary conditions using the
!          Ewald method. Only the contribution from the direct sum is
!          computed.
!
! Method: Ewald, "Die Berechnung optischer und elektrostatischer
!                 Gitterpotentiale", Annals. Phys. 64, 253 (1921).
!
! Arguments:
!
!           EDIR - Portion of the Coulomb energy comming from the
!                  direct Ewald sum.
!           FTOT - Array of dimension (3,NAT) containing the total
!                  force on each atom.
!           VIR  - Array of dimension (3,3) containing the virial
!                  tensor with elements given by:
!                  S_ab = SUM_i Fia * Rib where Fia = ath
!                  component of the force on atom i.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions - DERFC
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE EWALD_DIR (EDIR,FTOT)

        USE SystemParam, ONLY : rat,q,vir,dlv,cut,beta,nat,lnpair,&
                              & nnpair,ipbc,icou,ircp

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(OUT) :: edir
        DOUBLE PRECISION, INTENT(INOUT) :: ftot(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m,m1,m2,m3

        DOUBLE PRECISION :: x1,x2,x3,a1,a2,a3,qjk
        DOUBLE PRECISION :: s11,s12,s13,s22,s23,s33
        DOUBLE PRECISION :: a,b,d,c1,c2,xsq,arg

        DOUBLE PRECISION, PARAMETER :: fac = 1.128379167095512558561d0


        edir = 0.0d0

        IF ( icou == 0 ) RETURN
        IF ( nnpair == 0 ) RETURN


        IF ( ircp == 0 ) THEN

          a = beta * cut
          c1 = DERFC(a) / cut

          a = a * a
          c2 = beta * DEXP(-a)
          c2 = ( c1 + fac * c2 ) / cut

        ENDIF

        !=== Self Energy ===!

        DO i=1,nat
        edir = edir - q(i) * q(i)
        ENDDO

        a = fac * beta / 2.0d0
        edir = a * edir


        !=== Direct Ewald Sum ===!

        DO i=1,nnpair

          j = lnpair(1,i)
          k = lnpair(2,i)
          m = lnpair(3,i)

          m2 = MODULO(m,1000000)
          m3 = MODULO(m2,1000)
          m1 = m / 1000000
          m2 = m2 / 1000

          IF ( m1 > 100 ) m1 = 100 - m1
          IF ( m2 > 100 ) m2 = 100 - m2
          IF ( m3 > 100 ) m3 = 100 - m3

          a1 = DBLE(m1) * dlv(1,1) + DBLE(m2) * dlv(1,2) &
           & + DBLE(m3) * dlv(1,3)
          a2 = DBLE(m1) * dlv(2,1) + DBLE(m2) * dlv(2,2) &
           & + DBLE(m3) * dlv(2,3)
          a3 = DBLE(m1) * dlv(3,1) + DBLE(m2) * dlv(3,2) &
           & + DBLE(m3) * dlv(3,3)

          x1 = rat(1,j) - rat(1,k) - a1
          x2 = rat(2,j) - rat(2,k) - a2
          x3 = rat(3,j) - rat(3,k) - a3

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)

          qjk = q(j) * q(k)

          arg = beta * d
          a = DERFC(arg) / d

          edir = edir + qjk * a

          IF ( ircp == 0 ) THEN

            b = c2 * ( d - cut ) - c1

            edir = edir + qjk * b

          ENDIF

          arg = arg * arg
          b = beta * DEXP(-arg)

          a = ( a + fac * b ) / xsq

          IF ( ircp == 0 ) THEN

            a = a - c2 / d

          ENDIF

          a = qjk * a

          a1 = a * x1
          a2 = a * x2
          a3 = a * x3

          ftot(1,j) = ftot(1,j) + a1
          ftot(2,j) = ftot(2,j) + a2
          ftot(3,j) = ftot(3,j) + a3

          ftot(1,k) = ftot(1,k) - a1
          ftot(2,k) = ftot(2,k) - a2
          ftot(3,k) = ftot(3,k) - a3

          IF ( ipbc /= 2 ) CYCLE

          s11 = x1 * a1
          s12 = x1 * a2
          s13 = x1 * a3
          s22 = x2 * a2 
          s23 = x2 * a3
          s33 = x3 * a3

          vir(1,1) = vir(1,1) + s11
          vir(2,1) = vir(2,1) + s12
          vir(3,1) = vir(3,1) + s13
          vir(1,2) = vir(1,2) + s12
          vir(2,2) = vir(2,2) + s22
          vir(3,2) = vir(3,2) + s23
          vir(1,3) = vir(1,3) + s13
          vir(2,3) = vir(2,3) + s23
          vir(3,3) = vir(3,3) + s33

        ENDDO

        RETURN

      END SUBROUTINE EWALD_DIR
