! ==============================================================================
! Subroutine: EWALD_REC (EREC,FTOT)
! 
! Purpose: Computes the electrostatic energy, forces, and stress tensor of
!          a system of charges with periodic boundary conditions using the
!          Ewald method. Only the contribution from the reciprocial sum
!          is computed.
!
! Method: Ewald, "Die Berechnung optischer und elektrostatischer
!                 Gitterpotentiale", Annals. Phys. 64, 253 (1921).
!
! Arguments:
!
!           EREC - Porition of the Coulomb energy comming from the
!                  reciprocial Ewald sum.
!           FTOT - Array of dimension (3,NAT) containing the force
!                  force on each atom.
!           VIR  - Array of dimension (3,3) containing the virial
!                  tensor with elements given by:
!                  S_ab = SUM_i Fia * Rib where Fia = ath
!                  component of the force on atom i.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - GETMYJOBS
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE EWALD_REC (EREC,FTOT)

        USE SystemParam, ONLY : rat,q,vir,rlv,beta,vol,mrec,nat,ipbc,&
                              & icou,mpijob,myproc,nproc

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(OUT) :: erec
        DOUBLE PRECISION, INTENT(INOUT) :: ftot(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m,is,ie
        INTEGER :: k1,k2,k3,m1,m2,m3

        DOUBLE PRECISION :: x1,x2,x3,g1,g2,g3,c1,c2,c3
        DOUBLE PRECISION :: s11,s12,s13,s22,s23,s33,gsq
        DOUBLE PRECISION :: x,e,arg,sr(nat),si(nat)

        DOUBLE PRECISION, PARAMETER :: pi = 3.14159265358979323846d0
        DOUBLE PRECISION, PARAMETER :: twopi = 6.28318530717958623199d0


        erec = 0.0d0

        IF ( icou == 0 ) RETURN


        !=== Setup ===!

        k1 = mrec(1)
        k2 = mrec(2)
        k3 = mrec(3)

        k = 2 * k1 + 1
        k = k * ( 2 * k2 + 1 )
        k = k * ( 2 * k3 + 1 )

        c1 = pi / beta
        c1 = c1 * c1

        c2 = twopi * vol
        c2 = 1.0d0 / c2

        c3 = 4.0d0 * pi

        CALL GETMYJOBS (is,ie,k,mpijob,myproc,nproc)


        !=== Reciprocal Sum ===!

        IF ( is == 0 ) RETURN

        DO i=is,ie

          m1 = 2 * k1 + 1
          m2 = 2 * k2 + 1

          j = i - 1
          k = m1 * m2

          m3 = j / k
          j = j - m3 * k

          m2 = j / m1
          m1 = j - m2 * m1

          m1 = m1 - k1
          m2 = m2 - k2
          m3 = m3 - k3

          m = IABS(m1) + IABS(m2) + IABS(m3)

          IF ( m == 0 ) CYCLE

          g1 = DBLE(m1) * rlv(1,1) + DBLE(m2) * rlv(2,1) &
           & + DBLE(m3) * rlv(3,1)
          g2 = DBLE(m1) * rlv(1,2) + DBLE(m2) * rlv(2,2) &
           & + DBLE(m3) * rlv(3,2)
          g3 = DBLE(m1) * rlv(1,3) + DBLE(m2) * rlv(2,3) &
           & + DBLE(m3) * rlv(3,3)

          gsq = g1 * g1 + g2 * g2 + g3 * g3

          x1 = 0.0d0
          x2 = 0.0d0

          DO j=1,nat

            arg = g1 * rat(1,j) + g2 * rat(2,j) + g3 * rat(3,j)
            arg = twopi * arg

            sr(j) = q(j) * DCOS(arg)
            si(j) = q(j) * DSIN(arg)

            x1 = x1 + sr(j)
            x2 = x2 + si(j)

          ENDDO

          arg = c1 * gsq

          arg = c2 * DEXP(-arg) / gsq 

          e = arg * ( x1 * x1 + x2 * x2 )

          erec = erec + e

          DO j=1,nat

            x = x1 * si(j) - x2 * sr(j)
            x = c3 * x * arg

            ftot(1,j) = ftot(1,j) + x * g1
            ftot(2,j) = ftot(2,j) + x * g2
            ftot(3,j) = ftot(3,j) + x * g3

          ENDDO

          IF ( ipbc /= 2 ) CYCLE

          x = 1.0d0 / gsq + c1
          x = 2.0d0 * x * e

          x1 = x * g1
          x2 = x * g2
          x3 = x * g3

          s11 = x1 * g1
          s12 = x1 * g2
          s13 = x1 * g3
          s22 = x2 * g2
          s23 = x2 * g3
          s33 = x3 * g3

          vir(1,1) = vir(1,1) - s11
          vir(2,1) = vir(2,1) - s12
          vir(3,1) = vir(3,1) - s13
          vir(1,2) = vir(1,2) - s12
          vir(2,2) = vir(2,2) - s22
          vir(3,2) = vir(3,2) - s23
          vir(1,3) = vir(1,3) - s13
          vir(2,3) = vir(2,3) - s23
          vir(3,3) = vir(3,3) - s33

        ENDDO

        IF ( ipbc == 2 ) THEN

          vir(1,1) = vir(1,1) + erec
          vir(2,2) = vir(2,2) + erec
          vir(3,3) = vir(3,3) + erec

        ENDIF

        RETURN

      END SUBROUTINE EWALD_REC
