! ==============================================================================
! Subroutine: EXCLUDED (ECEX,EVEX,FTOT)
! 
! Purpose: Computes the correction to the electrostatic energy, force
!          and stress tensor due to atoms seperated by one or two
!          covalent bonds (1-2 and 1-3 interactions).
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
!          The correction is to subtract the coulomb and van der Waals
!          electrostatic energy terms for atoms seperated by one or
!          two or covalent bonds (1-2 and 1-3 interactions):
!
!          EC ---> EC - Q'i * Q'j / R
!          EV ---> EV - A / R^12 + B / R^6
!
!          where EC and EV are the Coulomb and van der Waals energies.
!          NOTE: Q'i is the scaled charge Qi / SQRT(eps)
!
! Arguments:
!
!           ECEX  - Energy due to excluded Coulomb interactions.
!           EVEX  - Energy due to excluded van der Waals interactions
!           FTOT  - Array of dimension (3,NAT) containing the total
!                   force on each atom.
!           VIR   - Array of dimension (3,3) containing the virial
!                   tensor with elements given by:
!                   S_ab = SUM_i Fia * Rib where Fia = ath
!                   component of the force on atom i.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - SMOOTH
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE EXCLUDED (ECEX,EVEX,FTOT)

        USE SystemParam, ONLY : rat,q,vir,dvdw,cut,scut,lex,nat,nex,&
                              & ipbc,icou,ivdw,ismth

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(OUT) :: ecex,evex
        DOUBLE PRECISION, INTENT(INOUT) :: ftot(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k

        DOUBLE PRECISION :: x1,x2,x3,s11,s12,s13,s22,s23,s33
        DOUBLE PRECISION :: a1,a2,a3,a,b,c,d,e,xsq,eps,req
        DOUBLE PRECISION :: s,ds,dds


        ecex = 0.0d0
        evex = 0.0d0

        IF ( nex == 0 ) RETURN
        IF ( icou == 0 .and. ivdw == 0 ) RETURN

        DO i=1,nex

          a = 0.0d0
          b = 0.0d0
          c = 0.0d0
          s = 1.0d0
          e = 0.0d0

          j = lex(1,i)
          k = lex(2,i)

          x1 = rat(1,j) - rat(1,k)
          x2 = rat(2,j) - rat(2,k)
          x3 = rat(3,j) - rat(3,k)

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)

          IF ( ismth /= 0 .and. d > scut ) THEN

            CALL SMOOTH (d,scut,cut,s,ds,dds)

          ENDIF

          IF ( icou == 1 ) THEN

            c = q(j) * q(k) / d

            ecex = ecex - c * s

          ENDIF

          IF ( ivdw == 1 ) THEN

            req = dvdw(1,j) + dvdw(1,k)
            eps = dvdw(2,j) * dvdw(2,k)
            eps = DSQRT(eps)

            a = req / d

            a = a * a
            b = a * a
            a = a * b

            b = eps * a
            a = a * b
            b = 2.0d0 * b

            e = a - b

            evex = evex - e * s

          ENDIF

          a = 12.0d0 * a - 6.0d0 * b + c
          a = a / xsq

          IF ( ismth /= 0 .and. d > scut ) THEN

            c = e + c
            c = c / d

            a = a * s - c * ds

          ENDIF

          a1 = a * x1
          a2 = a * x2
          a3 = a * x3

          ftot(1,j) = ftot(1,j) - a1
          ftot(2,j) = ftot(2,j) - a2
          ftot(3,j) = ftot(3,j) - a3

          ftot(1,k) = ftot(1,k) + a1
          ftot(2,k) = ftot(2,k) + a2
          ftot(3,k) = ftot(3,k) + a3

          IF ( ipbc /= 2 ) CYCLE

          s11 = a1 * x1
          s12 = a1 * x2
          s13 = a1 * x3
          s22 = a2 * x2 
          s23 = a2 * x3
          s33 = a3 * x3

          vir(1,1) = vir(1,1) - s11
          vir(2,1) = vir(2,1) - s12
          vir(3,1) = vir(3,1) - s13
          vir(1,2) = vir(1,2) - s12
          vir(2,2) = vir(2,2) - s22
          vir(3,2) = vir(3,2) - s23
          vir(1,3) = vir(1,3) - s13
          vir(2,3) = vir(2,3) - s23
          vir(3,3) = vir(3,3) - s33

        ENDDO

        RETURN

      END SUBROUTINE EXCLUDED
