! ==============================================================================
! Subroutine: GBORN (ECOUL,EVDW,FTOT)
! 
! Purpose: Computes the free energy, and force due to electrostatic
!          intereactions within the generalized Born framework.
!
! Method:  (1) V. Tsui and D.A. Case, Biopolymers, 56 275 (2000).
!          (2) Hawkins et al., J. Phys. Chem., 100 19824 (1996).
!
!          The energy in the generalized Born model is:
!
!          E = EVAC - ( 1 - 1 / EPS ) * Qi * Qj / FGB
!
!          where FGB = [ R^2 + Ai * Aj * Exp (-R^2 / 4 * Ai * Aj ) ]^1/2
!          EVAC is the Coulomb + van der Waals energy in vacuum
!          and Ai and Aj are the effective Born radii of the atoms.
!          See EQ 9 in Hawkins et al., J. Phys. Chem., 100 19824 for
!          a formula which computes Ai.
!
! Arguments:
!
!           ECOUL - Energy due to Coulomb interactions.
!           EVDW  - Energy due to van der Waals interactions.
!           FTOT  - Array of dimension (3,NAT) containing the total
!                   force on each atom.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - SMOOTH
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE GBORN (ECOUL,EVDW,FTOT)

        USE SystemParam, ONLY : rat,dvdw,dlv,scut,cut,grot,diel,kappa,&
                              & q,gbparm,indgb,lnpair,nnpair,nat,ipbc,&
                              & isym,isim,ivdw,ismth

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(OUT) :: ecoul,evdw
        DOUBLE PRECISION, INTENT(INOUT) :: ftot(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m,m1,m2,m3,jtp,ktp

        DOUBLE PRECISION :: x1,x2,x3,a1,a2,a3,rj,rk,sj,sk
        DOUBLE PRECISION :: a,b,c,d,ec,ev,qjk,fgb,req,eps
        DOUBLE PRECISION :: xd,xsq,arg,s,ds,dds
        DOUBLE PRECISION :: ebr(nat),deda(nat)


        !=== GBorn Parameters ===!

        !gbparm(6,i) = Intrinsic Born radius!
        !gbparm(7,i) = Hawkins scale factor!

        ecoul = 0.0d0
        evdw = 0.0d0

        ebr = 0.0d0
        deda = 0.0d0


        !=== Calculate Effective Born Radii ===!

        DO i=1,nnpair

          j = lnpair(1,i)
          k = lnpair(2,i)
          m = lnpair(3,i)

          jtp = indgb(j)
          ktp = indgb(k)

          rj = gbparm(6,jtp)
          rk = gbparm(6,ktp)
          sj = gbparm(7,jtp)
          sk = gbparm(7,ktp)

          IF ( isym == 0 .or. m == 0 ) THEN

            m2 = MODULO(m,1000000)
            m3 = MODULO(m2,1000)
            m1 = m / 1000000
            m2 = m2 / 1000

            IF ( m1 > 100 ) m1 = 100 - m1
            IF ( m2 > 100 ) m2 = 100 - m2
            IF ( m3 > 100 ) m3 = 100 - m3

            a1 = DBLE(m1) * dlv(1,1) + DBLE(m2) * dlv(1,2) &
             & + DBLE(m3) * dlv(1,3)
            a2 = DBLE(m1) * dlv(2,1) + DBLE(m2) * dlv(2,2) &
             & + DBLE(m3) * dlv(2,3)
            a3 = DBLE(m1) * dlv(3,1) + DBLE(m2) * dlv(3,2) &
             & + DBLE(m3) * dlv(3,3)

            x1 = rat(1,j) - rat(1,k) - a1
            x2 = rat(2,j) - rat(2,k) - a2
            x3 = rat(3,j) - rat(3,k) - a3

          ELSE

            x1 = rat(1,j) - grot(1,1,m) * rat(1,k) &
             & - grot(1,2,m) * rat(2,k) &
             & - grot(1,3,m) * rat(3,k)

            x2 = rat(2,j) - grot(2,1,m) * rat(1,k) &
             & - grot(2,2,m) * rat(2,k) &
             & - grot(2,3,m) * rat(3,k)

            x3 = rat(3,j) - grot(3,1,m) * rat(1,k) &
             & - grot(3,2,m) * rat(2,k) &
             & - grot(3,3,m) * rat(3,k)

          ENDIF

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)
          xd = 4.0d0 * d

          !=== Born Term For Atom j ===!

          c = sk * rk
          a = d + c
          b = d - c

          IF ( rj < a ) THEN

            a1 = 1.0d0 / a

            IF ( rj > b ) THEN
              a2 = 1.0d0 / rj
            ELSE
              a2 = 1.0d0 / b
            ENDIF

            c = a1 / a2
            c = DLOG(c)

            arg = a * b
            arg = arg * ( a1 + a2 ) - xd
            arg = arg * ( a1 - a2 )
            arg = 0.5d0 * arg + c

            ebr(j) = ebr(j) - arg / xd

          ENDIF

          !=== Born Term For Atom k ===!

          IF ( isym == 1 .and. m /= 0 ) CYCLE

          c = sj * rj
          a = d + c
          b = d - c

          IF ( rk < a ) THEN

            a1 = 1.0d0 / a

            IF ( rk > b ) THEN
              a2 = 1.0d0 / rk
            ELSE
              a2 = 1.0d0 / b
            ENDIF

            c = a1 / a2
            c = DLOG(c)

            arg = a * b
            arg = arg * ( a1 + a2 ) - xd
            arg = arg * ( a1 - a2 )
            arg = 0.5d0 * arg + c

            ebr(k) = ebr(k) - arg / xd

          ENDIF

        ENDDO


        !=== Compute Electrostatic Energy and Force ===!

        DO i=1,nat

          !=== First Invert and Check Born Radii ===!

          jtp = indgb(i)
          rj = gbparm(6,jtp)

          ebr(i) = ebr(i) + 1.0d0 / rj
          ebr(i) = 1.0d0 / ebr(i)

          IF ( ebr(i) < rj ) ebr(i) = rj

          !=== Self Energy Terms ===!

          fgb = ebr(i)

          arg = kappa * fgb
          a1 = DEXP(-arg) / diel

          qjk = 0.5d0 * q(i) * q(i)
          qjk = qjk / fgb
          a1 = qjk * a1

          ecoul = ecoul - qjk + a1

          arg = 1.0d0 + arg
          arg = qjk - arg * a1
          arg = arg / fgb

          deda(i) = deda(i) + arg

        ENDDO


        DO i=1,nnpair

          a = 0.0d0
          b = 0.0d0
          s = 1.0d0
          ev = 0.0d0

          j = lnpair(1,i)
          k = lnpair(2,i)
          m = lnpair(3,i)

          qjk = q(j) * q(k)

          IF ( isym == 0 .or. m == 0 ) THEN

            m2 = MODULO(m,1000000)
            m3 = MODULO(m2,1000)
            m1 = m / 1000000
            m2 = m2 / 1000

            IF ( m1 > 100 ) m1 = 100 - m1
            IF ( m2 > 100 ) m2 = 100 - m2
            IF ( m3 > 100 ) m3 = 100 - m3

            a1 = DBLE(m1) * dlv(1,1) + DBLE(m2) * dlv(1,2) &
             & + DBLE(m3) * dlv(1,3)
            a2 = DBLE(m1) * dlv(2,1) + DBLE(m2) * dlv(2,2) &
             & + DBLE(m3) * dlv(2,3)
            a3 = DBLE(m1) * dlv(3,1) + DBLE(m2) * dlv(3,2) &
             & + DBLE(m3) * dlv(3,3)

            x1 = rat(1,j) - rat(1,k) - a1
            x2 = rat(2,j) - rat(2,k) - a2
            x3 = rat(3,j) - rat(3,k) - a3

          ELSE

            x1 = rat(1,j) - grot(1,1,m) * rat(1,k) &
             & - grot(1,2,m) * rat(2,k) &
             & - grot(1,3,m) * rat(3,k)

            x2 = rat(2,j) - grot(2,1,m) * rat(1,k) &
             & - grot(2,2,m) * rat(2,k) &
             & - grot(2,3,m) * rat(3,k)

            x3 = rat(3,j) - grot(3,1,m) * rat(1,k) &
             & - grot(3,2,m) * rat(2,k) &
             & - grot(3,3,m) * rat(3,k)

          ENDIF

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)

          IF ( ismth /= 0 .and. d > scut ) THEN

            CALL SMOOTH (d,scut,cut,s,ds,dds)

          ENDIF

          IF ( ivdw == 1 ) THEN

            req = dvdw(1,j) + dvdw(1,k)
            eps = dvdw(2,j) * dvdw(2,k)
            eps = DSQRT(eps)

            a = req / d

            a = a * a
            b = a * a
            a = a * b

            b = eps * a
            a = a * b
            b = 2.0d0 * b

            ev = ( a - b )

            evdw = evdw + ev * s

          ENDIF

          c = qjk / d

          arg = ebr(j) * ebr(k)
          a1 = 0.25d0 * xsq / arg
          a2 = DEXP(-a1)

          xd = xsq + arg * a2
          fgb = DSQRT(xd)
          qjk = qjk / fgb

          arg = kappa * fgb
          a3 = DEXP(-arg) / diel
          a3 = qjk * a3

          ec = c - qjk + a3

          ecoul = ecoul + ec * s

          a = 12.0d0 * a - 6.0d0 * b + c
          a = a / xsq

          arg = 1.0d0 + arg
          arg = qjk - arg * a3
          arg = arg / xd

          a3 = a2 * ( 1.0d0 + a1 )
          a2 = 1.0d0 - 0.25d0 * a2

          a = a - a2 * arg
          b = 0.50d0 * a3 * arg

          IF ( ismth /= 0 .and. d > scut ) THEN

            c = ec + ev
            c = c / d

            a = a * s - c * ds
            b = b * s

          ENDIF

          a1 = a * x1
          a2 = a * x2
          a3 = a * x3

          ftot(1,j) = ftot(1,j) + a1
          ftot(2,j) = ftot(2,j) + a2
          ftot(3,j) = ftot(3,j) + a3

          deda(j) = deda(j) + b * ebr(k)

          IF ( isym == 0 .or. m == 0 ) THEN

            ftot(1,k) = ftot(1,k) - a1
            ftot(2,k) = ftot(2,k) - a2
            ftot(3,k) = ftot(3,k) - a3

            deda(k) = deda(k) + b * ebr(j)

          ENDIF

        ENDDO


        !=== Forces From dE/dA Terms ===!

        IF ( isim == 0 ) RETURN

        DO i=1,nnpair

          j = lnpair(1,i)
          k = lnpair(2,i)
          m = lnpair(3,i)

          jtp = indgb(j)
          ktp = indgb(k)

          rj = gbparm(6,jtp)
          rk = gbparm(6,ktp)
          sj = gbparm(7,jtp)
          sk = gbparm(7,ktp)

          IF ( isym == 0 .or. m == 0 ) THEN

            m2 = MODULO(m,1000000)
            m3 = MODULO(m2,1000)
            m1 = m / 1000000
            m2 = m2 / 1000

            IF ( m1 > 100 ) m1 = 100 - m1
            IF ( m2 > 100 ) m2 = 100 - m2
            IF ( m3 > 100 ) m3 = 100 - m3

            a1 = DBLE(m1) * dlv(1,1) + DBLE(m2) * dlv(1,2) &
             & + DBLE(m3) * dlv(1,3)
            a2 = DBLE(m1) * dlv(2,1) + DBLE(m2) * dlv(2,2) &
             & + DBLE(m3) * dlv(2,3)
            a3 = DBLE(m1) * dlv(3,1) + DBLE(m2) * dlv(3,2) &
             & + DBLE(m3) * dlv(3,3)

            x1 = rat(1,j) - rat(1,k) - a1
            x2 = rat(2,j) - rat(2,k) - a2
            x3 = rat(3,j) - rat(3,k) - a3

          ELSE

            x1 = rat(1,j) - grot(1,1,m) * rat(1,k) &
             & - grot(1,2,m) * rat(2,k) &
             & - grot(1,3,m) * rat(3,k)

            x2 = rat(2,j) - grot(2,1,m) * rat(1,k) &
             & - grot(2,2,m) * rat(2,k) &
             & - grot(2,3,m) * rat(3,k)

            x3 = rat(3,j) - grot(3,1,m) * rat(1,k) &
             & - grot(3,2,m) * rat(2,k) &
             & - grot(3,3,m) * rat(3,k)

          ENDIF

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)
          xd = 2.0d0 * d


          !=== Forces From j Term ===!

          c = sk * rk
          a = d + c
          b = d - c

          IF ( rj < a ) THEN

            a1 = 1.0d0 / a

            IF ( rj > b ) THEN
              a2 = 1.0d0 / rj
            ELSE
              a2 = 1.0d0 / b
            ENDIF

            arg = xsq + c * c
            arg = arg * ( a1 + a2 )
            arg = arg * ( a1 - a2 )

            c = a1 / a2
            c = DLOG(c)

            arg = arg - 2.0d0 * c

            c = ebr(j) / xd
            c = c * c / xd

            arg = c * arg * deda(j)

            a1 = arg * x1
            a2 = arg * x2
            a3 = arg * x3

            ftot(1,j) = ftot(1,j) - a1
            ftot(2,j) = ftot(2,j) - a2
            ftot(3,j) = ftot(3,j) - a3

            IF ( isym == 0 .or. m == 0 ) THEN

              ftot(1,k) = ftot(1,k) + a1
              ftot(2,k) = ftot(2,k) + a2
              ftot(3,k) = ftot(3,k) + a3

            ENDIF

          ENDIF

          !=== Forces From k Term ===!

          c = sj * rj
          a = d + c
          b = d - c

          IF ( rk < a ) THEN

            a1 = 1.0d0 / a

            IF ( rk > b ) THEN
              a2 = 1.0d0 / rk
            ELSE
              a2 = 1.0d0 / b
            ENDIF

            arg = xsq + c * c
            arg = arg * ( a1 + a2 )
            arg = arg * ( a1 - a2 )

            c = a1 / a2
            c = DLOG(c)

            arg = arg - 2.0d0 * c

            c = ebr(k) / xd
            c = c * c / xd

            arg = c * arg * deda(k)

            a1 = arg * x1
            a2 = arg * x2
            a3 = arg * x3

            ftot(1,j) = ftot(1,j) - a1
            ftot(2,j) = ftot(2,j) - a2
            ftot(3,j) = ftot(3,j) - a3

            IF ( isym == 0 .or. m == 0 ) THEN

              ftot(1,k) = ftot(1,k) + a1
              ftot(2,k) = ftot(2,k) + a2
              ftot(3,k) = ftot(3,k) + a3

            ENDIF

          ENDIF

        ENDDO

        RETURN

      END SUBROUTINE GBORN
