! ==============================================================================
! Subroutine: SCALED (ECSC,EVSC,FTOT)
! 
! Purpose: Computes the correction to the electrostatic energy, force
!          and stress tensor due to atoms seperated by three covalent
!          bonds.
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
!          The correction is to replace the coulomb and van der Waals
!          electrostatic energy terms for atoms seperated by three
!          covalent bonds by a scaled term:
!
!          EC ---> EC + [1/SC - 1] * [Q'i * Q'j / R]
!          EV ---> EV + [1/SV - 1] * [A / R^12 - B / R^6]
!
!          where EC and EV are the Coulomb and van der Waals energies
!          and SC and SV are scaling factors (SC,SV >= 1).
!          NOTE: Q'i is the scaled charge Qi / SQRT(eps)
!
! Arguments:
!
!           ECSC  - Energy due to scaled Coulomb interactions.
!           EVSC  - Energy due to scaled van der Waals interactions.
!           FTOT  - Array of dimension (3,NAT) containing the total
!                   force on each atom.
!           VIR   - Array of dimension (3,3) containing the virial
!                   tensor with elements given by:
!                   S_ab = SUM_i Fia * Rib where Fia = ath
!                   component of the force on atom i.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - SMOOTH
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE SCALED (ECSC,EVSC,FTOT)

        USE SystemParam, ONLY : rat,q,vir,dvdw,sfc,sfv,cut,scut,l14,&
                              & nat,n14,ipbc,icou,ivdw,ismth

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(OUT) :: ecsc,evsc
        DOUBLE PRECISION, INTENT(INOUT) :: ftot(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k

        DOUBLE PRECISION :: x1,x2,x3,s11,s12,s13,s22,s23,s33
        DOUBLE PRECISION :: a1,a2,a3,sc,sv,xsq,eps,req
        DOUBLE PRECISION :: a,b,c,d,e,s,ds,dds


        ecsc = 0.0d0
        evsc = 0.0d0

        IF ( n14 == 0 ) RETURN
        IF ( icou == 0 .and. ivdw == 0 ) RETURN

        sc = 1.0d0 / sfc - 1.0d0
        sv = 1.0d0 / sfv - 1.0d0

        DO i=1,n14

          a = 0.0d0
          b = 0.0d0
          c = 0.0d0
          s = 1.0d0
          e = 0.0d0

          j = l14(1,i)
          k = l14(2,i)

          x1 = rat(1,j) - rat(1,k)
          x2 = rat(2,j) - rat(2,k)
          x3 = rat(3,j) - rat(3,k)

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)

          IF ( ismth /= 0 .and. d > scut ) THEN

            CALL SMOOTH (d,scut,cut,s,ds,dds)

          ENDIF

          IF ( icou == 1 ) THEN

            c = q(j) * q(k) / d
            c = sc * c

            ecsc = ecsc + c * s

          ENDIF

          IF ( ivdw == 1 ) THEN

            req = dvdw(1,j) + dvdw(1,k)
            eps = dvdw(2,j) * dvdw(2,k)
            eps = DSQRT(eps)

            a = req / d

            a = a * a
            b = a * a
            a = a * b

            b = eps * a
            a = a * b
            b = 2.0d0 * b

            a = sv * a
            b = sv * b

            e = a - b

            evsc = evsc + e * s

          ENDIF

          a = 12.0d0 * a - 6.0d0 * b + c
          a = a / xsq

          IF ( ismth /= 0 .and. d > scut ) THEN

            c = e + c
            c = c / d

            a = a * s - c * ds

          ENDIF

          a1 = a * x1
          a2 = a * x2
          a3 = a * x3

          ftot(1,j) = ftot(1,j) + a1
          ftot(2,j) = ftot(2,j) + a2
          ftot(3,j) = ftot(3,j) + a3

          ftot(1,k) = ftot(1,k) - a1
          ftot(2,k) = ftot(2,k) - a2
          ftot(3,k) = ftot(3,k) - a3

          IF ( ipbc /= 2 ) CYCLE

          s11 = a1 * x1
          s12 = a1 * x2
          s13 = a1 * x3
          s22 = a2 * x2 
          s23 = a2 * x3
          s33 = a3 * x3

          vir(1,1) = vir(1,1) + s11
          vir(2,1) = vir(2,1) + s12
          vir(3,1) = vir(3,1) + s13
          vir(1,2) = vir(1,2) + s12
          vir(2,2) = vir(2,2) + s22
          vir(3,2) = vir(3,2) + s23
          vir(1,3) = vir(1,3) + s13
          vir(2,3) = vir(2,3) + s23
          vir(3,3) = vir(3,3) + s33

        ENDDO

        RETURN

      END SUBROUTINE SCALED
