! ==============================================================================
! Subroutine: VDW (EVDW,FTOT)
! 
! Purpose: Computes the energy, force and stress tensor due to van
!          der Waals interaction between two atoms in a periodic system.
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
!          The energy due to van der Waals is:
!
!          V(R) = A / R^12 - B / R^6
!
!          where A = EPS * ( REQ )**12 and B = 2 * EPS * ( REQ )**6
!          REQ is given by the sum of the van der Waals raddi for
!          each atom and EPS is the well depth.
!
! Arguments:
!
!           EVDW  - Energy due to van der Waals interactions.
!           FTOT  - Array of dimension (3,NAT) containing the total
!                   force on each atom.
!           VIR   - Array of dimension (3,3) containing the virial
!                   tensor with elements given by:
!                   S_ab = SUM_i Fia * Rib where Fia = ath
!                   component of the force on atom i.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - SMOOTH
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE VDW (EVDW,FTOT)

        USE SystemParam, ONLY : rat,vir,grot,dvdw,dlv,cut,scut,nat,&
                              & lnpair,nnpair,ipbc,isym,ivdw,ismth

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(OUT) :: evdw
        DOUBLE PRECISION, INTENT(INOUT) :: ftot(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m,m1,m2,m3

        DOUBLE PRECISION :: x1,x2,x3,a1,a2,a3,s11,s12,s13
        DOUBLE PRECISION :: s22,s23,s33,a,b,c,d,e,xsq,req
        DOUBLE PRECISION :: eps,s,ds,dds


        evdw = 0.0d0

        IF ( ivdw == 0 ) RETURN
        IF ( nnpair == 0 ) RETURN


        DO i=1,nnpair

          s = 1.0d0

          j = lnpair(1,i)
          k = lnpair(2,i)
          m = lnpair(3,i)

          req = dvdw(1,j) + dvdw(1,k)
          eps = dvdw(2,j) * dvdw(2,k)
          eps = DSQRT(eps)

          IF ( isym == 0 .or. m == 0 ) THEN

            m2 = MODULO(m,1000000)
            m3 = MODULO(m2,1000)
            m1 = m / 1000000
            m2 = m2 / 1000

            IF ( m1 > 100 ) m1 = 100 - m1
            IF ( m2 > 100 ) m2 = 100 - m2
            IF ( m3 > 100 ) m3 = 100 - m3

            a1 = DBLE(m1) * dlv(1,1) + DBLE(m2) * dlv(1,2) &
             & + DBLE(m3) * dlv(1,3)
            a2 = DBLE(m1) * dlv(2,1) + DBLE(m2) * dlv(2,2) &
             & + DBLE(m3) * dlv(2,3)
            a3 = DBLE(m1) * dlv(3,1) + DBLE(m2) * dlv(3,2) &
             & + DBLE(m3) * dlv(3,3)

            x1 = rat(1,j) - rat(1,k) - a1
            x2 = rat(2,j) - rat(2,k) - a2
            x3 = rat(3,j) - rat(3,k) - a3

          ELSE

            x1 = rat(1,j) - grot(1,1,m) * rat(1,k) &
             & - grot(1,2,m) * rat(2,k) &
             & - grot(1,3,m) * rat(3,k)

            x2 = rat(2,j) - grot(2,1,m) * rat(1,k) &
             & - grot(2,2,m) * rat(2,k) &
             & - grot(2,3,m) * rat(3,k)

            x3 = rat(3,j) - grot(3,1,m) * rat(1,k) &
             & - grot(3,2,m) * rat(2,k) &
             & - grot(3,3,m) * rat(3,k)

          ENDIF

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)

          IF ( ismth /= 0 .and. d > scut ) THEN

            CALL SMOOTH (d,scut,cut,s,ds,dds)

          ENDIF

          a = req / d

          a = a * a
          b = a * a
          a = a * b

          b = eps * a
          a = a * b
          b = 2.0d0 * b

          e = a - b

          evdw = evdw + e * s

          c = 12.0d0 * a - 6.0d0 * b
          c = c / xsq

          IF ( ismth /= 0 .and. d > scut ) THEN

            e = e / d
            c = c * s - e * ds

          ENDIF

          a1 = c * x1
          a2 = c * x2
          a3 = c * x3

          ftot(1,j) = ftot(1,j) + a1
          ftot(2,j) = ftot(2,j) + a2
          ftot(3,j) = ftot(3,j) + a3

          ftot(1,k) = ftot(1,k) - a1
          ftot(2,k) = ftot(2,k) - a2
          ftot(3,k) = ftot(3,k) - a3

          IF ( ipbc /= 2 ) CYCLE

          s11 = x1 * a1
          s12 = x1 * a2
          s13 = x1 * a3
          s22 = x2 * a2 
          s23 = x2 * a3
          s33 = x3 * a3

          vir(1,1) = vir(1,1) + s11
          vir(2,1) = vir(2,1) + s12
          vir(3,1) = vir(3,1) + s13
          vir(1,2) = vir(1,2) + s12
          vir(2,2) = vir(2,2) + s22
          vir(3,2) = vir(3,2) + s23
          vir(1,3) = vir(1,3) + s13
          vir(2,3) = vir(2,3) + s23
          vir(3,3) = vir(3,3) + s33

        ENDDO

        RETURN

      END SUBROUTINE VDW
