! ==============================================================================
! Subroutine: GDYNAMICALU (U,DU,SM) [ENM MPI VERSION]
! 
! Purpose: Computes the symmetry related dynamical matrix times
!          a vector.
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
! Arguments:
!
!           U  - Array of dimension (nbas*nat) containing the vector
!                that will operate on the symmetry related Dynamical
!                Matrix.
!           DU - Array of dimension (nbas*nat) containing the vector
!                that results from the operation D*U.
!           SM - Array of dimension (NAT) containing the inverse
!                square root of the mass of each atom.
!
! Special Notes:
!
!          (1)   Symmetry basis vectors are stored in the array GBAS
!                which has dimensions gbas(i,ib,ir) with 
!                i = 1,3*NSITE --- ib = 1,NBAS --- ir = 1,NROW
!
!          (2)   U(ia) is in symmetry coordinates and is stored using
!                the following scheme. U(ia) ranges from ia = 1,NBAS*NAT
!                and the first ia=1,NBAS components of U(ia) give the
!                coefficents of the basis vectors GBAS(i,ib,ir) that
!                one can use to construct a vector for atom 1 at all
!                NSITE "sites" or "cells" of the symmetric molecule. The
!                next NBAS components of U(ia) ia = NBAS+1,2*NBAS give
!                the coefficents that can be used to construct a vector
!                for atom 2 at all NSITE sites.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE GDYNAMICALU (U,DU,SM)

        USE SystemParam, ONLY : rat,grot,gbas,hnpair,lnpair,nnpair,&
                              & mgprm,nsite,nbas,nrow,nat,icou,ivdw

        IMPLICIT NONE

        INCLUDE 'mpif.h'

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(IN) :: u(nbas*nat),sm(nat)
        DOUBLE PRECISION, INTENT(INOUT) :: du(nbas*nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m,is,ir,ib,idx,jdx
        INTEGER :: j1,j2,j3,k1,k2,k3,ierr

        DOUBLE PRECISION :: x1,x2,x3,h11,h12,h13,h22
        DOUBLE PRECISION :: u1,u2,u3,h23,h33,s11,s12
        DOUBLE PRECISION :: a,b,d,de,dde,xsq,s22,wrk(nbas*nat)
        DOUBLE PRECISION :: dup(3*nat,nrow),up(3*nat,nrow)


        !=== Initialize ===!

        du = 0.0d0
        wrk = 0.0d0


        !=== Construct a Vector in Primative Cell ===!

        DO ir=1,nrow

          DO i=1,3*nat

            up(i,ir) = 0.0d0
            dup(i,ir) = 0.0d0

          ENDDO

          DO ib=1,nbas

            x1 = gbas(1,ib,ir)
            x2 = gbas(2,ib,ir)
            x3 = gbas(3,ib,ir)

            idx = 0
            jdx = ib

            DO i=1,nat

              up(idx+1,ir) = up(idx+1,ir) + x1 * u(jdx)
              up(idx+2,ir) = up(idx+2,ir) + x2 * u(jdx)
              up(idx+3,ir) = up(idx+3,ir) + x3 * u(jdx)

              idx = idx + 3
              jdx = jdx + nbas

            ENDDO

          ENDDO

        ENDDO


        !=== ENM Terms ===!

        IF ( nnpair == 0 ) GOTO 1

        DO i=1,nnpair

          j = lnpair(1,i)
          k = lnpair(2,i)
          m = lnpair(3,i)

          de = hnpair(1,i)
          dde = hnpair(2,i)

          IF ( m == 0 ) THEN

            is = 1

            x1 = rat(1,j) - rat(1,k)
            x2 = rat(2,j) - rat(2,k)
            x3 = rat(3,j) - rat(3,k)

          ELSE

            is = mgprm(m)

            x1 = rat(1,j) - grot(1,1,m) * rat(1,k) &
             & - grot(1,2,m) * rat(2,k) &
             & - grot(1,3,m) * rat(3,k)

            x2 = rat(2,j) - grot(2,1,m) * rat(1,k) &
             & - grot(2,2,m) * rat(2,k) &
             & - grot(2,3,m) * rat(3,k)

            x3 = rat(3,j) - grot(3,1,m) * rat(1,k) &
             & - grot(3,2,m) * rat(2,k) &
             & - grot(3,3,m) * rat(3,k)

          ENDIF

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)

          b = de / d
          a = dde - b
          a = a / xsq


          !=== Calculate D * U ===!

          j1 = 3 * j - 2
          j2 = j1 + 1
          j3 = j1 + 2

          IF ( is == 1 ) THEN

            k1 = 3 * k - 2
            k2 = k1 + 1
            k3 = k1 + 2

          ELSE

            jdx = nbas * ( k - 1 )

            k1 = 3 * is - 2
            k2 = k1 + 1
            k3 = k1 + 2

          ENDIF

          s11 = + sm(j) * sm(j)
          s12 = - sm(j) * sm(k)
          s22 = + sm(k) * sm(k)

          h11 = a * x1
          h22 = a * x2
          h33 = a * x3

          h12 = h11 * x2
          h13 = h11 * x3
          h23 = h22 * x3

          h11 = h11 * x1 + b
          h22 = h22 * x2 + b
          h33 = h33 * x3 + b

          IF ( is == 1 ) THEN

            DO ir=1,nrow

              u1 = s11 * up(j1,ir) + s12 * up(k1,ir)
              u2 = s11 * up(j2,ir) + s12 * up(k2,ir)
              u3 = s11 * up(j3,ir) + s12 * up(k3,ir)

              x1 = h11 * u1 + h12 * u2 + h13 * u3
              x2 = h12 * u1 + h22 * u2 + h23 * u3
              x3 = h13 * u1 + h23 * u2 + h33 * u3

              dup(j1,ir) = dup(j1,ir) + x1
              dup(j2,ir) = dup(j2,ir) + x2
              dup(j3,ir) = dup(j3,ir) + x3

              u1 = s12 * up(j1,ir) + s22 * up(k1,ir)
              u2 = s12 * up(j2,ir) + s22 * up(k2,ir)
              u3 = s12 * up(j3,ir) + s22 * up(k3,ir)

              x1 = h11 * u1 + h12 * u2 + h13 * u3
              x2 = h12 * u1 + h22 * u2 + h23 * u3
              x3 = h13 * u1 + h23 * u2 + h33 * u3

              dup(k1,ir) = dup(k1,ir) + x1
              dup(k2,ir) = dup(k2,ir) + x2
              dup(k3,ir) = dup(k3,ir) + x3

            ENDDO

          ELSE

            DO ir=1,nrow

              u1 = s11 * up(j1,ir)
              u2 = s11 * up(j2,ir)
              u3 = s11 * up(j3,ir)

              x1 = h11 * u1 + h12 * u2 + h13 * u3
              x2 = h12 * u1 + h22 * u2 + h23 * u3
              x3 = h13 * u1 + h23 * u2 + h33 * u3

              dup(j1,ir) = dup(j1,ir) + x1
              dup(j2,ir) = dup(j2,ir) + x2
              dup(j3,ir) = dup(j3,ir) + x3

              u1 = s12 * up(j1,ir)
              u2 = s12 * up(j2,ir)
              u3 = s12 * up(j3,ir)

              x1 = h11 * u1 + h12 * u2 + h13 * u3
              x2 = h12 * u1 + h22 * u2 + h23 * u3
              x3 = h13 * u1 + h23 * u2 + h33 * u3

              DO ib=1,nbas

                d = gbas(k1,ib,ir) * x1
                d = d + gbas(k2,ib,ir) * x2
                d = d + gbas(k3,ib,ir) * x3

                wrk(jdx+ib) = wrk(jdx+ib) + d

              ENDDO

            ENDDO

          ENDIF

        ENDDO

 1      CONTINUE


        a = DBLE(nsite) / DBLE(nrow)

        DO i=1,nat

          idx = 3 * ( i - 1 )
          jdx = nbas * ( i - 1 )

          DO ir=1,nrow
          DO ib=1,nbas

            d = gbas(1,ib,ir) * dup(idx+1,ir)
            d = d + gbas(2,ib,ir) * dup(idx+2,ir)
            d = d + gbas(3,ib,ir) * dup(idx+3,ir)

            wrk(jdx+ib) = wrk(jdx+ib) + d

          ENDDO
          ENDDO

          !=== Normalize ===!

          DO ib=1,nbas

            wrk(jdx+ib) = a * wrk(jdx+ib)

          ENDDO

        ENDDO

        j = nbas * nat

        CALL MPI_ALLREDUCE (wrk,du,j,mpi_double_precision,&
                          & mpi_sum,mpi_comm_world,ierr)

        RETURN

      END SUBROUTINE GDYNAMICALU
