! ==============================================================================
! Subroutine: ENERGY (EBOND,EANGL,EDIHD,ECOUL,EVDW,FTOT) [ISRS MPI VERSION]
! 
! Purpose: Computes the TOTAL energy, force and stress tensor due to
!          a classical molecular dynamics force field (e.g. AMBER).
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
! Arguments:
!
!           EBOND - Energy due to covalent bond stretching.
!           EANGL - Energy due to angle bending between two
!                   covalent bonds.
!           EDIHD - Energy due to dihedral rotations about
!                   covalent bonds.
!           ECOUL - Energy due to Coulomb interactions.
!           EVDW  - Energy due to van der Waals interactions.
!           FTOT  - Array of dimension (3,NAT) containing the total
!                   force on each atom.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - BOND, ANGLE, DIHEDRAL, GBORN, ECAVITY, ESTATIC,
!               EWALD_DIR, EWALD_REC, EWALD_PM, VDW, EXCLUDED,
!               SCALED, POLF
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE ENERGY (EBOND,EANGL,EDIHD,ECOUL,EVDW,FTOT)

        USE SystemParam, ONLY : vir,nat,ipbc,ircp,igb,igbsa

        IMPLICIT NONE

        INCLUDE 'mpif.h'

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(OUT) :: ebond,eangl,edihd,ecoul
        DOUBLE PRECISION, INTENT(OUT) :: evdw,ftot(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,ierr
        DOUBLE PRECISION :: wrk(3*nat+14),ans(3*nat+14)
        DOUBLE PRECISION :: edir,erec,ecav,ecex,evex,ecsc,evsc


        ebond = 0.0d0
        eangl = 0.0d0
        edihd = 0.0d0
        ecoul = 0.0d0
        evdw = 0.0d0

        edir = 0.0d0
        erec = 0.0d0
        ecav = 0.0d0
        ecex = 0.0d0
        evex = 0.0d0
        ecsc = 0.0d0
        evsc = 0.0d0

        ftot = 0.0d0
        vir = 0.0d0


        !=== Bond Terms ===!

        CALL BOND (ebond,ftot)

        !=== Angle Terms ===!

        CALL ANGLE (eangl,ftot)

        !=== Dihedral Terms ===!

        CALL DIHEDRAL (edihd,ftot)


        !=== Non-Bonded Terms ===!

        !=== Generalized Born Model ===!

        IF ( igb /= 0 ) THEN

          CALL GBORN (ecoul,evdw,ftot)

          CALL ECAVITY (ecav,ftot)

          ecoul = ecoul + ecav

        ENDIF

        !=== No Periodic Boundary ===!

        IF ( igb == 0 .and. ipbc == 0 ) THEN

          CALL ESTATIC (ecoul,evdw,ftot)

        ENDIF

        !=== Periodic Boundary ===!

        IF ( igb == 0 .and. ipbc /= 0 ) THEN

          CALL EWALD_DIR (edir,ftot)

          IF ( ircp == 1 ) THEN

            CALL EWALD_REC (erec,ftot)

          ELSEIF ( ircp == 2 ) THEN

            CALL EWALD_PM (erec,ftot)

          ENDIF

          ecoul = edir + erec

          CALL VDW (evdw,ftot)

        ENDIF

        !=== Non-Bonded Corrections ===!

        CALL EXCLUDED (ecex,evex,ftot)

        CALL SCALED (ecsc,evsc,ftot)

        !=== ISRS Force ===!

        CALL POLF (ftot)

        ecoul = ecoul + ecex + ecsc

        evdw = evdw + evex + evsc


        !=== Perform MPI Sum ===!

        !=== Package ===!

        j = 1

        DO i=1,nat

          wrk(j) = ftot(1,i)
          wrk(j+1) = ftot(2,i)
          wrk(j+2) = ftot(3,i)

          j = j + 3

        ENDDO

        DO i=1,3

          wrk(j) = vir(1,i)
          wrk(j+1) = vir(2,i)
          wrk(j+2) = vir(3,i)

          j = j + 3

        ENDDO

        wrk(j) = ebond
        wrk(j+1) = eangl
        wrk(j+2) = edihd
        wrk(j+3) = ecoul
        wrk(j+4) = evdw

        j = 3 * nat + 14

        ans = 0.0d0

        !=== Sum ===!

        CALL MPI_ALLREDUCE (wrk,ans,j,mpi_double_precision,&
                          & mpi_sum,mpi_comm_world,ierr)

        !=== Unpackage ===!

        j = 1

        DO i=1,nat

          ftot(1,i) = ans(j)
          ftot(2,i) = ans(j+1)
          ftot(3,i) = ans(j+2)

          j = j + 3

        ENDDO

        DO i=1,3

          vir(1,i) = ans(j)
          vir(2,i) = ans(j+1)
          vir(3,i) = ans(j+2)

          j = j + 3

        ENDDO

        ebond = ans(j)
        eangl = ans(j+1)
        edihd = ans(j+2)
        ecoul = ans(j+3)
        evdw = ans(j+4)

        RETURN

      END SUBROUTINE ENERGY
