! ==============================================================================
! Subroutine: POLF (FTOT)
! 
! Purpose: Computes the force on a bond due to Impulsive stimulate
!          Raman Scattering (ISRS) with visible light.
!
! Method:  Uses the bond polarizability model of Cardona and Snoke.
!          See E.C. Dykeman, D. Benson, O.F. Sankey and K.T. Tsen,
!          Phys. Rev. E, 80, 041909 (2009).
!
!          The ith component of the force on a bond due to ISRS is:
!
!          Fi = (1/4) * [fd * di + fe * Ei] * I * exp[-t^2 / Tau^2]
!
!          where Ei = ith component of the normailized E field vector
!                di = ith component of the normailized bond direction
!                Tau = pulse width of the light
!                I = intensity of light
!                a = alpha parallel
!                b = alpha perpendicular
!
!          fd and fe depend on the polarizabilty of the bond and are:
!
!          fd = (E.d)^2 [a' - b' - 2( a - b ) / d ] + b'
!          fe = (E.d) 2( a - b ) / d
!
! Arguments:
!
!           FTOT  - Array of dimension (3,NAT) containing the total
!                   force on each atom.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Special Notes:
!
!               The polariazation force takes the basic form:
!
!               F = (8*pi/c) * (alpha') * I/4 * Exp(-t^2/Tau^2)
!
!               I = Intensity in Watts/m^2
!               c = 3x10^8 m/s
!               alpha' = m^2 IN GAUSSIAN UNITS
!               cfac = (8*PI/C)*10^-15!
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE POLF (FTOT)

        USE SystemParam, ONLY : rat,lubd,nat,nubd,time,&
                              & taul,evec,xint,alpha

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(INOUT) :: ftot(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k

        DOUBLE PRECISION :: x1,x2,x3,c,d,fd,fe

        DOUBLE PRECISION, PARAMETER :: cfac = 5.2346856279d0


        IF ( nubd == 0 ) RETURN

        !=== Get exponent part ===!

        c = time / taul
        c = c * c

        c = 0.250d0 * xint * DEXP(-c)

        DO i=1,nubd

          j = lubd(1,i)
          k = lubd(2,i)

          x1 = rat(1,j) - rat(1,k)
          x2 = rat(2,j) - rat(2,k)
          x3 = rat(3,j) - rat(3,k)

          d = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(d)

          x1 = x1 / d
          x2 = x2 / d
          x3 = x3 / d

          !=== Form Dot Product ===!

          d = x1 * evec(1) + x2 * evec(2) + x3 * evec(3)

          fe = 2.0d0 * alpha(1) * d

          fd = d * ( alpha(2) - alpha(3) )
          fd = d * ( fd - fe ) + alpha(3)

          !=== Multiply by 1/4*I*Exp(-t^2/Tau^2) ===!

          fd = c * fd
          fe = c * fe

          !=== Unit Conversion to ev/Ang ===!

          !cfac = (8*PI/C)*10^-15!

          fd = cfac * fd
          fe = cfac * fe

          x1 = fd * x1 + fe * evec(1)
          x2 = fd * x2 + fe * evec(2)
          x3 = fd * x3 + fe * evec(3)

          ftot(1,j) = ftot(1,j) + x1
          ftot(2,j) = ftot(2,j) + x2
          ftot(3,j) = ftot(3,j) + x3

          ftot(1,k) = ftot(1,k) - x1
          ftot(2,k) = ftot(2,k) - x2
          ftot(3,k) = ftot(3,k) - x3

        ENDDO

        RETURN

      END SUBROUTINE POLF
