! ==============================================================================
! Program: SAGAURO [ISRS VERSION]
! 
! Description: A suite of algorithms for the simulation of biomolecules
!              and biomolecular complexes within a classical regime.
!
! Notes:
!        EXECUTION PATH
!
!        Section 1 - Read In Files and Prepare
!          - get arguments from command line
!          - call getoptions to get options for simulation
!          - call getparams to get coordinates and data
!          - open appropriate files
!          - setup system parameters
!        Section 2 - Perform the simulation
!          - call appropriate algorithm to perform
!            0) Minimize - Performs Minimization
!            1) MDynamics - Performs Molecular Dynamics
!            2) Vibes - Performs a Vibrational Analysis
!        Section 3 - Clean Up and Finish Program
!          - Close files
!          - Output Misc. Info if Ness.
!        END
!
!        FILE TREE
!
!        Unit = 1  --- Options File
!        Unit = 2  --- Parameter File
!        Unit = 3  --- Coordinate File
!        Unit = 4  --- Output File
!        Unit = 5  --- Standard Out (NOT USED)
!        Unit = 6  --- Standard In (NOT USED)
!        Unit = 7  --- Energy File
!        Unit = 8  --- Velocity File
!        Unit = 9  --- Trajectory/Eigenvector File
!        Unit = 10 --- Restart File
!        Unit = 11 --- Molecule File
!        Unit = 12 --- Group Symmetry File
!        Unit = 13 --- Group Character File
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - GETOPTIONS, CHKOPTIONS, GETPARAMS, GETGTPARAMS, NEIGHBORMAP
!               SYMMETRYMAP, EWALDBETA, EWALDSETUP, PMESETUP, MINIMIZE,
!               MDYNAMICS, VIBES, SETUPMPI, CLOSEMPI
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      PROGRAM SAGUARO

        USE SystemParam

        IMPLICIT NONE

        !=== VARIABLES ===!

        INTEGER :: i,iargc,narg

        CHARACTER (LEN=80) :: infile,gf2kfile,parmfile,molfile,arg
        CHARACTER (LEN=80) :: outfile,efile,vfile,cfile,symfile,ctfile


        !=== START PROGRAM ===!

        !=== SECTION 1 ===!

        NAMELIST /options/ cut,scut,bxgrd,diel,kappa,tole,tols,tolf,&
                         & sfc,sfv,beta,temp,btemp,taut,taup,pres,comp,&
                         & tstep,nsteps,nse,nsv,nsc,nsmap,nsbk,nev,&
                         & maxsi,nord,isim,icrd,ipbc,isym,irep,ircp,&
                         & iout,imin,itc,ipc,ips,igb,igbsa,ishk,ibd,&
                         & iag,idh,icou,ivdw,iseed,ismth

        !=== Setup MPI ===!

        CALL SETUPMPI

        !=== Default File Names ===!

        infile = 'sys.opt'
        outfile = 'output'
        gf2kfile = 'sys.gf2k'
        parmfile = 'sys.parm'
        molfile = 'sys.mol'
        efile = 'energy'
        vfile = 'velocity'
        cfile = 'trajectory'
        rstfile = 'sys.rst'
        symfile = 'sys.sym'
        ctfile = 'sys.CT'

        narg = IARGC ()

        DO i=1,narg,2

          CALL GETARG (i,arg)

          SELECT CASE (arg)

            CASE ('-i')
              CALL GETARG (i+1,infile)
            CASE ('-c')
              CALL GETARG (i+1,gf2kfile)
            CASE ('-p')
              CALL GETARG (i+1,parmfile)
            CASE ('-o')
              CALL GETARG (i+1,outfile)
            CASE ('-m')
              CALL GETARG (i+1,molfile)
            CASE ('-e')
              CALL GETARG (i+1,efile)
            CASE ('-v')
              CALL GETARG (i+1,vfile)
            CASE ('-x')
              CALL GETARG (i+1,cfile)
            CASE ('-r')
              CALL GETARG (i+1,rstfile)
            CASE ('-s')
              CALL GETARG (i+1,symfile)
            CASE ('-t')
              CALL GETARG (i+1,ctfile)
            CASE DEFAULT

              IF ( myproc == 0 ) WRITE(*,*)arg,'Invalid Line Argument'

              IF ( mpijob ) CALL CLOSEMPI

              STOP

          END SELECT

        ENDDO


        !=== READ IN EXTRA DATA FOR ISRS ===!

        OPEN (UNIT = 99,FILE = 'extra.dat',STATUS = 'UNKNOWN')

        READ(99,*,END=1)evec(1),evec(2),evec(3)
        READ(99,*)xint,taul
        READ(99,*)alpha(1)
        READ(99,*)alpha(2)
        READ(99,*)alpha(3)

        CLOSE (UNIT = 99)
        GOTO 2

 1      WRITE(*,*)'ERROR: The file extra.dat does not exist.'
        WRITE(*,*)'This file contains settings for ISRS.'
        WRITE(*,*)'Please create this file.'

        STOP

 2      CONTINUE

        !=== Open Input Files ===!

        OPEN (UNIT = 1,FILE = infile,STATUS = 'OLD')
        OPEN (UNIT = 2,FILE = parmfile,STATUS = 'OLD')
        OPEN (UNIT = 3,FILE = gf2kfile,STATUS = 'OLD')
        OPEN (UNIT = 11,FILE = molfile,STATUS = 'OLD')

        IF ( myproc == 0 ) THEN

          OPEN (UNIT = 4,FILE = outfile,STATUS = 'UNKNOWN')

        ENDIF

        !=== Get Options ===!

        CALL GETOPTIONS

        CALL CHKOPTIONS

        !=== Get Parameters ===!

        CALL GETPARAMS

        !=== Close Input Files ===!

        CLOSE (UNIT = 1)
        CLOSE (UNIT = 2)
        CLOSE (UNIT = 3)
        CLOSE (UNIT = 11)

        !=== Open Output Files ===!

        IF ( myproc == 0 ) THEN

          OPEN (UNIT = 10,FILE = rstfile,STATUS = 'UNKNOWN')

        ENDIF

        SELECT CASE (isim)

          CASE (0)

            IF ( myproc == 0 ) THEN
            OPEN (UNIT = 7,FILE = efile,STATUS = 'UNKNOWN')
            ENDIF

          CASE (1)

            IF ( nse >= 0 .and. myproc == 0 ) THEN
            OPEN (UNIT = 7,FILE = efile,STATUS = 'UNKNOWN')
            ENDIF

            IF ( nsv >= 0 .and. myproc == 0 ) THEN
            OPEN (UNIT = 8,FILE = vfile,STATUS = 'UNKNOWN')
            ENDIF

            IF ( nsc >= 0 .and. myproc == 0 ) THEN
            OPEN (UNIT = 9,FILE = cfile,STATUS = 'UNKNOWN')
            ENDIF

          CASE (2)

            IF ( myproc == 0 ) THEN
            OPEN (UNIT = 7,FILE = efile,STATUS = 'UNKNOWN')
            OPEN (UNIT = 9,FILE = cfile,STATUS = 'UNKNOWN')
            ENDIF

          CASE DEFAULT

            IF ( myproc == 0 ) THEN
            WRITE(4,*)'ERROR: isim must range from 0 to 2'
            ENDIF

            IF ( mpijob ) CALL CLOSEMPI

            STOP

          END SELECT

        ENDIF


        !=== WELCOME ===!

        IF ( myproc == 0 ) THEN

          WRITE(4,*)'**************************************************'
          WRITE(4,*)'*                    WELCOME                     *'
          WRITE(4,*)'*                      TO                        *'
          WRITE(4,*)'*                    SAUGARO                     *'
          WRITE(4,*)'**************************************************'
          WRITE(4,*)' '

          WRITE(4,*)'WRITTEN BY: Eric C. Dykeman and Otto F. Sankey'
          WRITE(4,*)'            Department of Physics and Astronomy'
          WRITE(4,*)'            Arizona State University'
          WRITE(4,*)'            Tempe, AZ 85287-1504'
          WRITE(4,*)' '
          WRITE(4,*)'FILE STATEMENTS'
          WRITE(4,*)'Input File:  ',infile
          WRITE(4,*)'Output File: ',outfile
          WRITE(4,*)'Param File:  ',parmfile
          WRITE(4,*)'Coord File:  ',gf2kfile
          WRITE(4,*)'Rst File:    ',rstfile
          WRITE(4,*)'Traj File:   ',cfile
          WRITE(4,*)'Velc File:   ',vfile
          WRITE(4,*)'Ener File:   ',efile
          WRITE(4,*)'Sym File:    ',symfile
          WRITE(4,*)'CT File:     ',ctfile
          WRITE(4,*)' '

          WRITE(4,*)'INPUT FILE (OPTIONS)'
          WRITE(4,NML=options)
          WRITE(4,*)' '

        ENDIF

        !=== Check Boundary Conditions ===!

        SELECT CASE (ipbc)

          CASE (0)

            IF ( myproc == 0 ) THEN
            WRITE(4,*)'SAGUARO: system has no periodicity'
            ENDIF

          CASE (1)

            IF ( myproc == 0 ) THEN
            WRITE(4,*)'SAGUARO: system has constant volume'
            ENDIF

          CASE (2)

            IF ( myproc == 0 ) THEN
            WRITE(4,*)'SAGUARO: system has constant pressure'
            ENDIF

          CASE DEFAULT

            IF ( myproc == 0 ) THEN
            WRITE(4,*)'ERROR: ipbc must range from 0 to 2'
            ENDIF

            IF ( mpijob ) CALL CLOSEMPI

            STOP

        END SELECT

        !=== Setup Generalized Born ===!

        IF ( igb =/ 0 ) THEN

          IF ( myproc == 0 ) THEN
          WRITE(4,*)'SAGUARO: using the generalized Born model'
          ENDIF

          CALL SETUPGB

        ENDIF

        !=== Setup Symmetry ===!

        IF ( isym /= 0 ) THEN

          OPEN (UNIT = 12,FILE = symfile,STATUS = 'OLD')
          OPEN (UNIT = 13,FILE = ctfile,STATUS = 'OLD')

          IF ( myproc == 0 ) THEN
          WRITE(4,*)'SAGUARO: applying group theory'
          ENDIF

          CALL GETGTPARAMS

          CALL SETUPGT

          CLOSE (UNIT = 12)
          CLOSE (UNIT = 13)

        ENDIF

        !=== Setup Ewald Sum ===!

        IF ( ipbc /= 0 .and. igb == 0 ) THEN

          SELECT CASE (ircp)

            CASE (0)

              IF ( myproc == 0 ) THEN
              WRITE(4,*)'SAGUARO: using the shifted potential method'
              ENDIF

            CASE (1)

              IF ( myproc == 0 ) THEN
              WRITE(4,*)'SAGUARO: using a standard Ewald sum'
              ENDIF

              CALL EWALDBETA

              CALL EWALDSETUP

            CASE (2)

              IF ( myproc == 0 ) THEN
              WRITE(4,*)'SAGUARO: using the particle mesh method'
              ENDIF

              CALL EWALDBETA

              CALL PMESETUP

            CASE DEFAULT

              IF ( myproc == 0 ) THEN
              WRITE(4,*)'ERROR: ircp must range from 0 to 2'
              ENDIF

              IF ( mpijob ) CALL CLOSEMPI

              STOP

          END SELECT

        ENDIF

        !=== Setup Inital NeighborMap ===!

        IF ( isym == 0 ) THEN

          CALL NEIGHBORMAP

        ELSEIF ( isym == 1 ) THEN

          CALL SYMMETRYMAP

        ENDIF


        !=== SECTION 2 ===!

        SELECT CASE (isim)

          CASE (0)

            IF ( myproc == 0 ) THEN
            WRITE(4,*)'SAGUARO: performing minimization'
            ENDIF

            CALL MINIMIZE

          CASE (1)

            IF ( myproc == 0 ) THEN
            WRITE(4,*)'SAGUARO: performing molecular dynamics'
            ENDIF

            CALL MDYNAMICS

          CASE (2)

            IF ( myproc == 0 ) THEN
            WRITE(4,*)'SAGUARO: performing a vibrational analysis'
            ENDIF

            CALL VIBES

          CASE DEFAULT

            IF ( myproc == 0 ) THEN
            WRITE(4,*)'ERROR: isim must range from 0 to 2'
            ENDIF

            IF ( mpijob ) CALL CLOSEMPI

            STOP

        END SELECT


        !=== SECTION 3 ===!

        IF ( myproc == 0 ) THEN

          WRITE(4,*)'INFO: quantities are in the following units'
          WRITE(4,*)' '
          WRITE(4,*)'ENERGY            Electron Volts'
          WRITE(4,*)'FORCES            Electron Volts / Angstrom'
          WRITE(4,*)'ACCELERATIONS     Angstroms / (PicoSecond)^2'
          WRITE(4,*)'VELOCITY          Angstroms / PicoSecond'
          WRITE(4,*)'DISTANCE          Angstroms'
          WRITE(4,*)'TIME              PicoSeconds'

        ENDIF

        !=== Close Files ===!

        IF ( myproc == 0 ) THEN

          CLOSE (UNIT = 4)
          CLOSE (UNIT = 10)

        ENDIF

        SELECT CASE (isim)

          CASE (0)

            IF ( myproc == 0 ) CLOSE (UNIT = 7)

          CASE (1)

            IF ( myproc == 0 ) THEN
            IF ( nse /= 0 ) CLOSE (UNIT = 7)
            IF ( nsv /= 0 ) CLOSE (UNIT = 8)
            IF ( nsc /= 0 ) CLOSE (UNIT = 9)
            ENDIF

          CASE (2)

            IF ( myproc == 0 ) THEN
            CLOSE (UNIT = 7)
            CLOSE (UNIT = 9)
            ENDIF

          CASE DEFAULT

            IF ( myproc == 0 ) THEN
            WRITE(4,*)'ERROR: isim must range from 0 to 2'
            ENDIF

            IF ( mpijob ) CALL CLOSEMPI

            STOP

        END SELECT

        IF ( mpijob ) CALL CLOSEMPI

      END PROGRAM SAGAURO
