! ==============================================================================
! Subroutine: ANGLEDU (U,DU,SM)
! 
! Purpose: Computes the contribution to the dynmaical matrix times a
!          vector from angle bending between two covalent bonds.
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
!          The energy due to angle bending is:
!
!          E = E0 * (T - TEQ)^2
!
!          where E0 = 1/2 the Spring Constant, and TEQ is the
!          equilibrium angle between the two covalent bonds.
!
! Arguments:
!
!           U  - Array of dimension (3*NAT) containing the vector
!                that will operate on the Dynamical Matrix.
!           DU - Array of dimension (3*NAT) containing the vector
!                that results from the operation D*U.
!           SM - Array of dimension (NAT) containing the inverse
!                square root of the mass of each atom.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE ANGLEDU (U,DU,SM)

        USE SystemParam, ONLY : rat,dangl,langl,nat,nag,iag

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(IN) :: u(3*nat),sm(nat)
        DOUBLE PRECISION, INTENT(INOUT) :: du(3*nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m,j1,j2,j3
        INTEGER :: k1,k2,k3,m1,m2,m3

        DOUBLE PRECISION :: x1,x2,x3,y1,y2,y3,a1,a2,a3,b1,b2,b3
        DOUBLE PRECISION :: h11,h12,h13,h21,h22,h23,h31,h32,h33
        DOUBLE PRECISION :: u1,u2,u3,w1,w2,w3,s11,s12,s13,s22,s23
        DOUBLE PRECISION :: a,b,c,d,d1,d2,t,teq,spr,xsq,ysq,s33

        DOUBLE PRECISION, PARAMETER :: tm = 0.999d0


        IF ( iag == 0 .or. nag == 0 ) RETURN


        DO i=1,nag

          j = langl(1,i)
          k = langl(2,i)
          m = langl(3,i)

          spr = dangl(1,i)
          teq = dangl(2,i)

          x1 = rat(1,j) - rat(1,k)
          x2 = rat(2,j) - rat(2,k)
          x3 = rat(3,j) - rat(3,k)

          y1 = rat(1,m) - rat(1,k)
          y2 = rat(2,m) - rat(2,k)
          y3 = rat(3,m) - rat(3,k)

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          ysq = y1 * y1 + y2 * y2 + y3 * y3

          d = x1 * y1 + x2 * y2 + x3 * y3

          d2 = xsq * ysq
          d1 = DSQRT(d2)

          c = d / d1

          c = MIN(+tm,c)
          c = MAX(-tm,c)

          t = DACOS(c)

          b = t - teq
          a = 2.0d0 * spr

          t = DSIN(t)
          b = a * b / t

          a = a - b * c
          c = t * t
          a = a / c

          a = a / d2
          b = b / d1

          d1 = d / xsq
          d2 = d / ysq

          a1 = y1 - d1 * x1
          a2 = y2 - d1 * x2
          a3 = y3 - d1 * x3

          b1 = x1 - d2 * y1
          b2 = x2 - d2 * y2
          b3 = x3 - d2 * y3


          !=== Calculate D * U ===!

          j1 = 3 * j - 2
          j2 = j1 + 1
          j3 = j1 + 2

          k1 = 3 * k - 2
          k2 = k1 + 1
          k3 = k1 + 2

          m1 = 3 * m - 2 
          m2 = m1 + 1
          m3 = m1 + 2

          s11 = + sm(j) * sm(j)
          s12 = - sm(j) * sm(k)
          s13 = + sm(j) * sm(m)
          s22 = + sm(k) * sm(k)
          s23 = - sm(k) * sm(m)
          s33 = + sm(m) * sm(m)

          !=== Term 1 ===!

          c = b / xsq

          h11 = a * a1 + c * x1
          h22 = a * a2 + c * x2
          h33 = a * a3 + c * x3

          w1 = c * ( a1 - d1 * x1 )
          w2 = c * ( a2 - d1 * x2 )
          w3 = c * ( a3 - d1 * x3 )

          c = c * d

          h12 = h11 * a2 + w1 * x2
          h13 = h11 * a3 + w1 * x3
          h23 = h22 * a3 + w2 * x3

          h11 = h11 * a1 + w1 * x1 + c
          h22 = h22 * a2 + w2 * x2 + c
          h33 = h33 * a3 + w3 * x3 + c

          u1 = s11 * u(j1) + s12 * u(k1)
          u2 = s11 * u(j2) + s12 * u(k2)
          u3 = s11 * u(j3) + s12 * u(k3)

          w1 = h11 * u1 + h12 * u2 + h13 * u3
          w2 = h12 * u1 + h22 * u2 + h23 * u3
          w3 = h13 * u1 + h23 * u2 + h33 * u3

          du(j1) = du(j1) + w1
          du(j2) = du(j2) + w2
          du(j3) = du(j3) + w3

          u1 = s12 * u(j1) + s22 * u(k1)
          u2 = s12 * u(j2) + s22 * u(k2)
          u3 = s12 * u(j3) + s22 * u(k3)

          w1 = h11 * u1 + h12 * u2 + h13 * u3
          w2 = h12 * u1 + h22 * u2 + h23 * u3
          w3 = h13 * u1 + h23 * u2 + h33 * u3

          du(k1) = du(k1) + w1
          du(k2) = du(k2) + w2
          du(k3) = du(k3) + w3

          !=== Term 2 ===!

          h11 = a * b1 + b * y1 / ysq
          h22 = a * b2 + b * y2 / ysq
          h33 = a * b3 + b * y3 / ysq

          w1 = b * x1 / xsq
          w2 = b * x2 / xsq
          w3 = b * x3 / xsq

          h12 = h11 * a2 + w1 * x2
          h13 = h11 * a3 + w1 * x3
          h23 = h22 * a3 + w2 * x3

          h21 = h22 * a1 + w2 * x1
          h31 = h33 * a1 + w3 * x1
          h32 = h33 * a2 + w3 * x2

          h11 = h11 * a1 + w1 * x1 - b
          h22 = h22 * a2 + w2 * x2 - b
          h33 = h33 * a3 + w3 * x3 - b

          w1 = h11 * u1 + h12 * u2 + h13 * u3
          w2 = h21 * u1 + h22 * u2 + h23 * u3
          w3 = h31 * u1 + h32 * u2 + h33 * u3

          du(k1) = du(k1) + w1
          du(k2) = du(k2) + w2
          du(k3) = du(k3) + w3

          u1 = s13 * u(j1) + s23 * u(k1)
          u2 = s13 * u(j2) + s23 * u(k2)
          u3 = s13 * u(j3) + s23 * u(k3)

          w1 = h11 * u1 + h12 * u2 + h13 * u3
          w2 = h21 * u1 + h22 * u2 + h23 * u3
          w3 = h31 * u1 + h32 * u2 + h33 * u3

          du(m1) = du(m1) + w1
          du(m2) = du(m2) + w2
          du(m3) = du(m3) + w3

          !=== Term 3 ===!

          w1 = h12
          w2 = h13
          w3 = h23

          h12 = h21
          h13 = h31
          h23 = h32

          h21 = w1
          h31 = w2
          h32 = w3

          u1 = s12 * u(k1) + s13 * u(m1)
          u2 = s12 * u(k2) + s13 * u(m2)
          u3 = s12 * u(k3) + s13 * u(m3)

          w1 = h11 * u1 + h12 * u2 + h13 * u3
          w2 = h21 * u1 + h22 * u2 + h23 * u3
          w3 = h31 * u1 + h32 * u2 + h33 * u3

          du(j1) = du(j1) + w1
          du(j2) = du(j2) + w2
          du(j3) = du(j3) + w3

          u1 = s22 * u(k1) + s23 * u(m1)
          u2 = s22 * u(k2) + s23 * u(m2)
          u3 = s22 * u(k3) + s23 * u(m3)

          w1 = h11 * u1 + h12 * u2 + h13 * u3
          w2 = h21 * u1 + h22 * u2 + h23 * u3
          w3 = h31 * u1 + h32 * u2 + h33 * u3

          du(k1) = du(k1) + w1
          du(k2) = du(k2) + w2
          du(k3) = du(k3) + w3

          !=== Term 4 ===!

          c = b / ysq

          h11 = a * b1 + c * y1
          h22 = a * b2 + c * y2
          h33 = a * b3 + c * y3

          w1 = c * ( b1 - d2 * y1 )
          w2 = c * ( b2 - d2 * y2 )
          w3 = c * ( b3 - d2 * y3 )

          c = c * d

          h12 = h11 * b2 + w1 * y2
          h13 = h11 * b3 + w1 * y3
          h23 = h22 * b3 + w2 * y3

          h11 = h11 * b1 + w1 * y1 + c
          h22 = h22 * b2 + w2 * y2 + c
          h33 = h33 * b3 + w3 * y3 + c

          w1 = h11 * u1 + h12 * u2 + h13 * u3
          w2 = h12 * u1 + h22 * u2 + h23 * u3
          w3 = h13 * u1 + h23 * u2 + h33 * u3

          du(k1) = du(k1) + w1
          du(k2) = du(k2) + w2
          du(k3) = du(k3) + w3

          u1 = s23 * u(k1) + s33 * u(m1)
          u2 = s23 * u(k2) + s33 * u(m2)
          u3 = s23 * u(k3) + s33 * u(m3)

          w1 = h11 * u1 + h12 * u2 + h13 * u3
          w2 = h12 * u1 + h22 * u2 + h23 * u3
          w3 = h13 * u1 + h23 * u2 + h33 * u3

          du(m1) = du(m1) + w1
          du(m2) = du(m2) + w2
          du(m3) = du(m3) + w3

        ENDDO

        RETURN

      END SUBROUTINE ANGLEDU
