! ==============================================================================
! Subroutine: BONDDU (U,DU,SM)
! 
! Purpose: Computes the contribution to the dynmaical matrix times a
!          vector from bond stretching in a covalent bond.
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
!          The energy due to bond stretching is:
!
!          E = E0 * (D - DEQ)^2
!
!          where E0 = 1/2 the Spring Constant, and DEQ is the
!          equilibrium bond length between the two atoms.
!
! Arguments:
!
!           U  - Array of dimension (3*NAT) containing the vector
!                that will operate on the Dynamical Matrix.
!           DU - Array of dimension (3*NAT) containing the vector
!                that results from the operation D*U.
!           SM - Array of dimension (NAT) containing the inverse
!                square root of the mass of each atom.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE BONDDU (U,DU,SM)

        USE SystemParam, ONLY : rat,dubd,lubd,nat,nubd,ibd

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(IN) :: u(3*nat),sm(nat)
        DOUBLE PRECISION, INTENT(INOUT) :: du(3*nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k
        INTEGER :: j1,j2,j3,k1,k2,k3

        DOUBLE PRECISION :: x1,x2,x3,h11,h12,h13,h22
        DOUBLE PRECISION :: u1,u2,u3,h23,h33,s11,s12
        DOUBLE PRECISION :: a,b,d,dsq,deq,spr,s22


        IF ( ibd == 0 .or. nubd == 0 ) RETURN


        DO i=1,nubd

          j = lubd(1,i)
          k = lubd(2,i)

          spr = dubd(1,i)
          deq = dubd(2,i)

          x1 = rat(1,j) - rat(1,k)
          x2 = rat(2,j) - rat(2,k)
          x3 = rat(3,j) - rat(3,k)

          dsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(dsq)

          a = deq / d
          b = 2.0d0 * spr

          a = a * b
          b = b - a
          a = a / dsq


          !=== Calculate D * U ===!

          j1 = 3 * j - 2
          j2 = j1 + 1
          j3 = j1 + 2

          k1 = 3 * k - 2
          k2 = k1 + 1
          k3 = k1 + 2

          s11 = + sm(j) * sm(j)
          s12 = - sm(j) * sm(k)
          s22 = + sm(k) * sm(k)

          h11 = a * x1
          h22 = a * x2
          h33 = a * x3

          h12 = h11 * x2
          h13 = h11 * x3
          h23 = h22 * x3

          h11 = h11 * x1 + b
          h22 = h22 * x2 + b
          h33 = h33 * x3 + b

          u1 = s11 * u(j1) + s12 * u(k1)
          u2 = s11 * u(j2) + s12 * u(k2)
          u3 = s11 * u(j3) + s12 * u(k3)

          x1 = h11 * u1 + h12 * u2 + h13 * u3
          x2 = h12 * u1 + h22 * u2 + h23 * u3
          x3 = h13 * u1 + h23 * u2 + h33 * u3

          du(j1) = du(j1) + x1
          du(j2) = du(j2) + x2
          du(j3) = du(j3) + x3

          u1 = s12 * u(j1) + s22 * u(k1)
          u2 = s12 * u(j2) + s22 * u(k2)
          u3 = s12 * u(j3) + s22 * u(k3)

          x1 = h11 * u1 + h12 * u2 + h13 * u3
          x2 = h12 * u1 + h22 * u2 + h23 * u3
          x3 = h13 * u1 + h23 * u2 + h33 * u3

          du(k1) = du(k1) + x1
          du(k2) = du(k2) + x2
          du(k3) = du(k3) + x3

        ENDDO

        RETURN

      END SUBROUTINE BONDDU
