! ==============================================================================
! Subroutine: DIHEDRALDU (U,DU,SM)
!
! Purpose: Computes the contribution to the dynamical matrix times a
!          vector from dihedral rotation about a covalent bond. 
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
!          The energy due to a dihedral rotation is:
!
!          E = E0 * (1 + Cos[N*PHI - GAMMA])
!
!          where E0 = the Energy Constant, N is the dihedral multiplicity,
!          PHI is the dihedral angle, and GAMMA is the dihedral phase angle.
!
! Arguments:
!
!           U  - Array of dimension (3*NAT) containing the vector
!                that will operate on the Dynamical Matrix.
!           DU - Array of dimension (3*NAT) containing the vector
!                that results from the operation D*U.
!           SM - Array of dimension (NAT) containing the inverse
!                square root of the mass of each atom.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - CROSS
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE DIHEDRALDU (U,DU,SM)

        USE SystemParam, ONLY : rat,ddihd,ldihd,nat,ndh,idh

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(IN) :: u(3*nat),sm(nat)
        DOUBLE PRECISION, INTENT(INOUT) :: du(3*nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m,n,icnt

        DOUBLE PRECISION :: u1,u2,u3,w1,w2,w3,d1,d2,f1,f2,c,d,arg
        DOUBLE PRECISION :: asq,bsq,phi,spr,gam,pn,s11,s12,s21,s22
        DOUBLE PRECISION :: a(3),b(3),x(3),y(3),z(3),s(2,3),h(3,3)
        DOUBLE PRECISION :: xxa(3),xxb(3),yxa(3),d12(3),d32(3)
        DOUBLE PRECISION :: zxa(3),zxb(3),yxb(3),d43(3),dot(6)

        DOUBLE PRECISION, PARAMETER :: sml = 1.00d-6
        DOUBLE PRECISION, PARAMETER :: two = 2.00d0
        DOUBLE PRECISION, PARAMETER :: twopi = 6.28318530717958623199d0


        IF ( idh == 0 .or. ndh == 0 ) RETURN

        DO i=1,ndh

          j = ldihd(1,i)
          k = ldihd(2,i)
          m = ldihd(3,i)
          n = ldihd(4,i)

          spr = ddihd(1,i)
          gam = ddihd(2,i)
          pn  = ddihd(3,i)

          s(1,1) = +sm(j)
          s(2,1) = -sm(k)
          s(1,2) = -sm(k)
          s(2,2) = +sm(m)
          s(1,3) = -sm(m)
          s(2,3) = +sm(n)

          x(1) = rat(1,j) - rat(1,k)
          x(2) = rat(2,j) - rat(2,k)
          x(3) = rat(3,j) - rat(3,k)

          y(1) = rat(1,m) - rat(1,k)
          y(2) = rat(2,m) - rat(2,k)
          y(3) = rat(3,m) - rat(3,k)

          z(1) = rat(1,n) - rat(1,m)
          z(2) = rat(2,n) - rat(2,m)
          z(3) = rat(3,n) - rat(3,m)

          CALL CROSS (x,y,a)
          CALL CROSS (z,y,b)

          asq = a(1) * a(1) + a(2) * a(2) + a(3) * a(3)
          bsq = b(1) * b(1) + b(2) * b(2) + b(3) * b(3)

          d = a(1) * b(1) + a(2) * b(2) + a(3) * b(3)
          c = a(1) * z(1) + a(2) * z(2) + a(3) * z(3)

          IF ( asq < sml ) THEN
            asq = 0.0d0
          ELSE
            asq = 1.0d0 / asq
          ENDIF

          IF ( bsq < sml ) THEN
            bsq = 0.0d0
          ELSE
            bsq = 1.0d0 / bsq
          ENDIF

          d2 = asq * bsq
          d1 = DSQRT(d2)

          phi = d * d1

          phi = MIN(+1.0d0,phi)
          phi = MAX(-1.0d0,phi)

          phi = DACOS(phi)

          IF ( c > 0.0d0 ) phi = twopi - phi

          arg = pn * phi - gam

          f1 = pn * DCOS(arg)
          c = DSIN(phi)

          IF ( DABS(c) > sml ) THEN

            f2 = DSIN(arg) / c

            f1 = f2 * DCOS(phi) - f1
            f1 = f1 / ( c * c )

          ELSE

            f2 = f1 * DSIGN(1.0d0,DCOS(phi))

            f1 = f1 * ( pn * pn - 1.0d0 )
            f1 = f1 / 3.0d0

          ENDIF

          c = spr * pn

          f1 = f1 * c * d2
          f2 = f2 * c * d1

          d1 = d * asq
          d2 = d * bsq

          !=== Delta Functions ===!

          d12 = 0.0d0
          d32 = 0.0d0
          d43 = 0.0d0

          d12(1) = 1.0d0
          d32(2) = 1.0d0
          d43(3) = 1.0d0

          !=== Cross Products ===!

          CALL CROSS(x,a,xxa)
          CALL CROSS(x,b,xxb)
          CALL CROSS(y,a,yxa)
          CALL CROSS(y,b,yxb)
          CALL CROSS(z,a,zxa)
          CALL CROSS(z,b,zxb)

          !=== Dot Products ===!

          dot(1) = x(1) * x(1) + x(2) * x(2) + x(3) * x(3)
          dot(2) = y(1) * y(1) + y(2) * y(2) + y(3) * y(3)
          dot(3) = z(1) * z(1) + z(2) * z(2) + z(3) * z(3)
          dot(4) = x(1) * y(1) + x(2) * y(2) + x(3) * y(3)
          dot(5) = x(1) * z(1) + x(2) * z(2) + x(3) * z(3)
          dot(6) = y(1) * z(1) + y(2) * z(2) + y(3) * z(3)


          !=== Calculate D * U ===!

          DO icnt=1,9

            k = (icnt-1) / 3
            j = icnt - k * 3
            k = k + 1

            s11 = s(1,j) * s(1,k)
            s12 = s(1,j) * s(2,k)
            s21 = s(2,j) * s(1,k)
            s22 = s(2,j) * s(2,k)

            !=== Calculate Hessian ===!

            DO n=1,3
            DO m=1,3

              !=== First Part ===!

              u1 = - y(m) * y(n)
              u2 = two * z(m) * y(n) - y(m) * z(n)
              u3 = two * y(m) * z(n) - z(m) * y(n)

              w1 = d12(j) * ( d43(k) - d1 * d12(k) )
              w1 = w1 + d43(j) * ( d12(k) - d2 * d43(k) )
              w2 = d32(k) * ( d12(j) - d2 * d43(j) )
              w3 = d32(j) * ( d12(k) - d2 * d43(k) )

              IF ( m == n ) THEN
              u1 = u1 + dot(2)
              u2 = u2 - dot(6)
              u3 = u3 - dot(6)
              ENDIF

              c = u1 * w1 + u2 * w2 + u3 * w3

              u1 = d1 * x(m) * x(n) + d2 * z(m) * z(n)
              u1 = u1 - x(m) * z(n) - z(m) * x(n)
              u2 = two * x(m) * y(n) - y(m) * x(n)
              u3 = two * y(m) * x(n) - x(m) * y(n)

              w1 = d32(j) * d32(k)
              w2 = d32(k) * ( d43(j) - d1 * d12(j) )
              w3 = d32(j) * ( d43(k) - d1 * d12(k) )

              IF ( m == n ) THEN
              u1 = u1 + two * dot(5) 
              u1 = u1 - d1 * dot(1)
              u1 = u1 - d2 * dot(3)
              u2 = u2 - dot(4)
              u3 = u3 - dot(4)
              ENDIF

              c = c + u1 * w1 + u2 * w2 + u3 * w3

              !=== Second Part ===!

              w1 = yxb(m) * d12(j) + yxa(m) * d43(j)
              w1 = w1 - ( zxa(m) + xxb(m) ) * d32(j)

              w2 = asq * ( yxa(m) * d12(j) - xxa(m) * d32(j) )
              w3 = bsq * ( yxb(m) * d43(j) - zxb(m) * d32(j) )

              u1 = yxb(n) * d12(k) + yxa(n) * d43(k)
              u1 = u1 - ( zxa(n) + xxb(n) ) * d32(k)

              u2 = asq * ( yxa(n) * d12(k) - xxa(n) * d32(k) )
              u3 = bsq * ( yxb(n) * d43(k) - zxb(n) * d32(k) )

              w2 = w2 + w3
              w3 = w2 - two * w3
              w1 = w1 - d * w2

              u2 = u2 + u3
              u3 = u2 - two * u3
              u1 = u1 - d * u2

              c = c + d * w3 * u3 - w1 * u2 - u1 * w2
              h(m,n) = f1 * w1 * u1 + f2 * c

            ENDDO
            ENDDO

            m = k
            n = k + 1
            k = j + 1

            j = 3 * ldihd(j,i) - 3
            k = 3 * ldihd(k,i) - 3
            m = 3 * ldihd(m,i) - 3
            n = 3 * ldihd(n,i) - 3

            u1 = s11 * u(m+1) + s12 * u(n+1)
            u2 = s11 * u(m+2) + s12 * u(n+2)
            u3 = s11 * u(m+3) + s12 * u(n+3)

            w1 = h(1,1) * u1 + h(1,2) * u2 + h(1,3) * u3
            w2 = h(2,1) * u1 + h(2,2) * u2 + h(2,3) * u3
            w3 = h(3,1) * u1 + h(3,2) * u2 + h(3,3) * u3

            du(j+1) = du(j+1) + w1
            du(j+2) = du(j+2) + w2
            du(j+3) = du(j+3) + w3

            u1 = s21 * u(m+1) + s22 * u(n+1)
            u2 = s21 * u(m+2) + s22 * u(n+2)
            u3 = s21 * u(m+3) + s22 * u(n+3)

            w1 = h(1,1) * u1 + h(1,2) * u2 + h(1,3) * u3
            w2 = h(2,1) * u1 + h(2,2) * u2 + h(2,3) * u3
            w3 = h(3,1) * u1 + h(3,2) * u2 + h(3,3) * u3

            du(k+1) = du(k+1) + w1
            du(k+2) = du(k+2) + w2
            du(k+3) = du(k+3) + w3

          ENDDO

        ENDDO

        RETURN

      END SUBROUTINE DIHEDRALDU
