! ==============================================================================
! Subroutine: DYNAMICALU (U,DU,SM) [MPI VERSION]
! 
! Purpose: Computes the dynamical matrix times a vector.
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
! Arguments:
!
!           U  - Array of dimension (3*NAT) containing the vector
!                that will operate on the Dynamical Matrix.
!           DU - Array of dimension (3*NAT) containing the vector
!                that results from the operation D*U.
!           SM - Array of dimension (NAT) containing the inverse
!                square root of the mass of each atom.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - BONDDU, ANGLEDU, DIHEDRALDU
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE DYNAMICALU (U,DU,SM)

        USE SystemParam, ONLY : rat,dlv,hnpair,nat,lnpair,nnpair,&
                              & icou,ivdw

        IMPLICIT NONE

        INCLUDE 'mpif.h'

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(IN) :: u(3*nat),sm(nat)
        DOUBLE PRECISION, INTENT(INOUT) :: du(3*nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m,m1,m2,m3
        INTEGER :: j1,j2,j3,k1,k2,k3,ierr

        DOUBLE PRECISION :: x1,x2,x3,h11,h12,h13,h22
        DOUBLE PRECISION :: u1,u2,u3,h23,h33,s11,s12
        DOUBLE PRECISION :: a,b,d,de,dde,xsq,s22,wrk(3*nat)


        !=== Initialize ===!

        du = 0.0d0
        wrk = 0.0d0


        !=== Bond Terms ===!

        CALL BONDDU (u,wrk,sm)

        !=== Angle Terms ===!

        CALL ANGLEDU (u,wrk,sm)

        !=== Dihedral Terms ===!

        CALL DIHEDRALDU (u,wrk,sm)


        !=== Non-Bonded Terms ===!

        IF ( nnpair == 0 ) GOTO 1
        IF ( icou == 0 .and. ivdw == 0 ) GOTO 1

        DO i=1,nnpair

          j = lnpair(1,i)
          k = lnpair(2,i)
          m = lnpair(3,i)

          de = hnpair(1,i)
          dde = hnpair(2,i)

          m2 = MODULO(m,1000000)
          m3 = MODULO(m2,1000)
          m1 = m / 1000000
          m2 = m2 / 1000

          IF ( m1 > 100 ) m1 = 100 - m1
          IF ( m2 > 100 ) m2 = 100 - m2
          IF ( m3 > 100 ) m3 = 100 - m3

          u1 = DBLE(m1) * dlv(1,1) + DBLE(m2) * dlv(1,2) &
           & + DBLE(m3) * dlv(1,3)
          u2 = DBLE(m1) * dlv(2,1) + DBLE(m2) * dlv(2,2) &
           & + DBLE(m3) * dlv(2,3)
          u3 = DBLE(m1) * dlv(3,1) + DBLE(m2) * dlv(3,2) &
           & + DBLE(m3) * dlv(3,3)

          x1 = rat(1,j) - rat(1,k) - u1
          x2 = rat(2,j) - rat(2,k) - u2
          x3 = rat(3,j) - rat(3,k) - u3

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)

          b = de / d
          a = dde - b
          a = a / xsq


          !=== Calculate D * U ===!

          j1 = 3 * j - 2
          j2 = j1 + 1
          j3 = j1 + 2

          k1 = 3 * k - 2
          k2 = k1 + 1
          k3 = k1 + 2

          s11 = + sm(j) * sm(j)
          s12 = - sm(j) * sm(k)
          s22 = + sm(k) * sm(k)

          h11 = a * x1
          h22 = a * x2
          h33 = a * x3

          h12 = h11 * x2
          h13 = h11 * x3
          h23 = h22 * x3

          h11 = h11 * x1 + b
          h22 = h22 * x2 + b
          h33 = h33 * x3 + b

          u1 = s11 * u(j1) + s12 * u(k1)
          u2 = s11 * u(j2) + s12 * u(k2)
          u3 = s11 * u(j3) + s12 * u(k3)

          x1 = h11 * u1 + h12 * u2 + h13 * u3
          x2 = h12 * u1 + h22 * u2 + h23 * u3
          x3 = h13 * u1 + h23 * u2 + h33 * u3

          wrk(j1) = wrk(j1) + x1
          wrk(j2) = wrk(j2) + x2
          wrk(j3) = wrk(j3) + x3

          u1 = s12 * u(j1) + s22 * u(k1)
          u2 = s12 * u(j2) + s22 * u(k2)
          u3 = s12 * u(j3) + s22 * u(k3)

          x1 = h11 * u1 + h12 * u2 + h13 * u3
          x2 = h12 * u1 + h22 * u2 + h23 * u3
          x3 = h13 * u1 + h23 * u2 + h33 * u3

          wrk(k1) = wrk(k1) + x1
          wrk(k2) = wrk(k2) + x2
          wrk(k3) = wrk(k3) + x3

        ENDDO

 1      j = 3 * nat

        CALL MPI_ALLREDUCE (wrk,du,j,mpi_double_precision,&
                          & mpi_sum,mpi_comm_world,ierr)

        RETURN

      END SUBROUTINE DYNAMICALU
