! ==============================================================================
! Subroutine: ESTATICH
! 
! Purpose: Calculates the first and second derivatives of the Coulomb
!          and van der Waals potentials between atom pairs required
!          for the calculation of the hessian.
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
!          V(R) = A / R^12 - B / R^6 + Q'i * Q'j / R
!
!          NOTE: Q'i is the scaled charge Qi / SQRT(eps)
!
!          HNPAIR(1,i) = dV/dR
!          HNPAIR(2,i) = d2V/dR2
!
! Arguments:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - SMOOTH
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE ESTATICH

        USE SystemParam, ONLY : rat,q,dvdw,dlv,scut,cut,grot,isym,&
                              & hnpair,lnpair,nnpair,icou,ivdw,ismth

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m,m1,m2,m3

        DOUBLE PRECISION :: x1,x2,x3,a1,a2,a3,xsq
        DOUBLE PRECISION :: a,b,d,e,s,ds,dds,req,eps


        IF ( nnpair == 0 ) RETURN
        IF ( icou == 0 .and. ivdw == 0 ) RETURN


        DO i=1,nnpair

          j = lnpair(1,i)
          k = lnpair(2,i)
          m = lnpair(3,i)

          IF ( isym == 0 .or. m == 0 ) THEN

            m2 = MODULO(m,1000000)
            m3 = MODULO(m2,1000)
            m1 = m / 1000000
            m2 = m2 / 1000

            IF ( m1 > 100 ) m1 = 100 - m1
            IF ( m2 > 100 ) m2 = 100 - m2
            IF ( m3 > 100 ) m3 = 100 - m3

            a1 = DBLE(m1) * dlv(1,1) + DBLE(m2) * dlv(1,2) &
             & + DBLE(m3) * dlv(1,3)
            a2 = DBLE(m1) * dlv(2,1) + DBLE(m2) * dlv(2,2) &
             & + DBLE(m3) * dlv(2,3)
            a3 = DBLE(m1) * dlv(3,1) + DBLE(m2) * dlv(3,2) &
             & + DBLE(m3) * dlv(3,3)

            x1 = rat(1,j) - rat(1,k) - a1
            x2 = rat(2,j) - rat(2,k) - a2
            x3 = rat(3,j) - rat(3,k) - a3

          ELSE

            x1 = rat(1,j) - grot(1,1,m) * rat(1,k) &
             & - grot(1,2,m) * rat(2,k) &
             & - grot(1,3,m) * rat(3,k)

            x2 = rat(2,j) - grot(2,1,m) * rat(1,k) &
             & - grot(2,2,m) * rat(2,k) &
             & - grot(2,3,m) * rat(3,k)

            x3 = rat(3,j) - grot(3,1,m) * rat(1,k) &
             & - grot(3,2,m) * rat(2,k) &
             & - grot(3,3,m) * rat(3,k)

          ENDIF

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)

          a = 0.0d0
          b = 0.0d0
          e = 0.0d0

          IF ( icou == 1 ) THEN

            e = q(j) * q(k) / d

            a = -e / d
            b = 2.0d0 * e
            b = b / xsq

          ENDIF

          IF ( ivdw == 1 ) THEN

            req = dvdw(1,j) + dvdw(1,k)
            eps = dvdw(2,j) * dvdw(2,k)
            eps = DSQRT(eps)

            a1 = req / d

            a1 = a1 * a1
            a2 = a1 * a1
            a1 = a1 * a2

            a2 = eps * a1
            a1 = a1 * a2
            a2 = 2.0d0 * a2

            e = e + a1 - a2

            a1 = 12.0d0 * a1
            a2 = 6.0d0 * a2

            a = a + ( a2 - a1 ) / d

            a1 = 13.0d0 * a1
            a2 = 7.0d0 * a2

            b = b + ( a1 - a2 ) / xsq

          ENDIF

          IF ( ismth /= 0 .and. d > scut ) THEN

            CALL SMOOTH (d,scut,cut,s,ds,dds)

            x1 = a * s + e * ds
            x2 = 2.0d0 * a * ds
            x2 = b * s + e * dds + x2

            a = x1
            b = x2

          ENDIF

          hnpair(1,i) = hnpair(1,i) + a
          hnpair(2,i) = hnpair(2,i) + b

        ENDDO

        RETURN

      END SUBROUTINE ESTATICH
