! ==============================================================================
! Subroutine: EXCLUDEDH
! 
! Purpose: Calculates the correction to the first and second derivatives
!          of the Coulomb and van der Waals potentials between atom
!          pairs required for the calculation of the hessian. The
!          correction takes into account that the first and second
!          bonded nearest neighbors do not have electrostatic terms.
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
!          V(R) = -[ A / R^12 - B / R^6 + Q'i * Q'j / R ]
!
!          NOTE: Q'i is the scaled charge Qi / SQRT(eps)
!
!          HNPAIR(1,i) = dV/dR
!          HNPAIR(2,i) = d2V/dR2
!
! Arguments:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2009   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - SMOOTH
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE EXCLUDEDH

        USE SystemParam, ONLY : rat,q,dvdw,dlv,scut,cut,hnpair,&
                              & lnpair,lex,nnpair,nex,icou,ivdw,ismth

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m,icnt,ipair

        DOUBLE PRECISION :: x1,x2,x3,a1,a2,xsq
        DOUBLE PRECISION :: a,b,d,e,s,ds,dds,req,eps


        IF ( nex == 0 .or. nnpair == 0 ) RETURN
        IF ( icou == 0 .and. ivdw == 0 ) RETURN


        DO i=1,nex

          j = lex(1,i)
          k = lex(2,i)

          ipair = 0

          DO m=1,nnpair

            IF ( lnpair(3,m) /= 0 ) CYCLE

            icnt = 0

            IF ( lnpair(1,m) == j ) icnt = icnt + 1
            IF ( lnpair(2,m) == j ) icnt = icnt + 1
            IF ( lnpair(1,m) == k ) icnt = icnt + 1
            IF ( lnpair(2,m) == k ) icnt = icnt + 1

            IF ( icnt == 2 ) THEN

              ipair = m
              EXIT

            ENDIF

          ENDDO

          IF ( ipair == 0 ) CYCLE

          x1 = rat(1,j) - rat(1,k)
          x2 = rat(2,j) - rat(2,k)
          x3 = rat(3,j) - rat(3,k)

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)

          a = 0.0d0
          b = 0.0d0
          e = 0.0d0

          IF ( icou == 1 ) THEN

            e = q(j) * q(k) / d

            a = -e / d
            b = 2.0d0 * e
            b = b / xsq

          ENDIF

          IF ( ivdw == 1 ) THEN

            req = dvdw(1,j) + dvdw(1,k)
            eps = dvdw(2,j) * dvdw(2,k)
            eps = DSQRT(eps)

            a1 = req / d

            a1 = a1 * a1
            a2 = a1 * a1
            a1 = a1 * a2

            a2 = eps * a1
            a1 = a1 * a2
            a2 = 2.0d0 * a2

            e = e + a1 - a2

            a1 = 12.0d0 * a1
            a2 = 6.0d0 * a2

            a = a + ( a2 - a1 ) / d

            a1 = 13.0d0 * a1
            a2 = 7.0d0 * a2

            b = b + ( a1 - a2 ) / xsq

          ENDIF

          IF ( ismth /= 0 .and. d > scut ) THEN

            CALL SMOOTH (d,scut,cut,s,ds,dds)

            x1 = a * s + e * ds
            x2 = 2.0d0 * a * ds
            x2 = b * s + e * dds + x2

            a = x1
            b = x2

          ENDIF

          hnpair(1,ipair) = hnpair(1,ipair) - a
          hnpair(2,ipair) = hnpair(2,ipair) - b

        ENDDO

        RETURN

      END SUBROUTINE EXCLUDEDH
