! ==============================================================================
! Subroutine: GBORNH
! 
! Purpose: Calculates the first and second derivatives of the
!          Generalized Born potential between atom pairs for
!          calcualtion of the hessian.
!
! Method:  (1) V. Tsui and D.A. Case, Biopolymers, 56 275 (2000).
!          (2) Hawkins et al., J. Phys. Chem., 100 19824 (1996).
!
!          The energy in the generalized Born model is:
!
!          V(R) = EVAC - (1 - 1 / eps ) * Qi * Qj / FGB
!
!          where FGB = [ R^2 + Ai * Aj * Exp (-R^2 / 4 * Ai * Aj ) ]^1/2
!          EVAC is the Coulomb + van der Waals energy in vacuum and
!          Ai and Aj are the effective Born radii of the atoms. See
!          EQ 9 in Hawkins et al., J. Phys. Chem., 100 19824 for a
!          formula which computes Ai.
!
!          HNPAIR(1,i) = dV/dR
!          HNPAIR(2,i) = d2V/dR2
!
! Arguments:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - SMOOTH
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE GBORNH

        USE SystemParam, ONLY : rat,dvdw,dlv,scut,cut,grot,diel,kappa,&
                              & q,gbparm,indgb,hnpair,lnpair,nnpair,&
                              & nat,isym,ivdw,ismth

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        INTEGER :: i,j,k,m,m1,m2,m3,jtp,ktp

        DOUBLE PRECISION :: x1,x2,x3,a1,a2,a3,rj,rk,sj,sk
        DOUBLE PRECISION :: a,b,c,d,e,qjk,fgb,req,eps,xd
        DOUBLE PRECISION :: xsq,arg,s,ds,dds,ebr(nat)


        !=== GBorn Parameters ===!

        !gbparm(6,i) = Intrinsic Born radius!
        !gbparm(7,i) = Hawkins scale factor!

        ebr = 0.0d0


        !=== Get Effective Born Radii ===!

        DO i=1,nnpair

          j = lnpair(1,i)
          k = lnpair(2,i)
          m = lnpair(3,i)

          jtp = indgb(j)
          ktp = indgb(k)

          rj = gbparm(6,jtp)
          rk = gbparm(6,ktp)
          sj = gbparm(7,jtp)
          sk = gbparm(7,ktp)

          IF ( isym == 0 .or. m == 0 ) THEN

            m2 = MODULO(m,1000000)
            m3 = MODULO(m2,1000)
            m1 = m / 1000000
            m2 = m2 / 1000

            IF ( m1 > 100 ) m1 = 100 - m1
            IF ( m2 > 100 ) m2 = 100 - m2
            IF ( m3 > 100 ) m3 = 100 - m3

            a1 = DBLE(m1) * dlv(1,1) + DBLE(m2) * dlv(1,2) &
             & + DBLE(m3) * dlv(1,3)
            a2 = DBLE(m1) * dlv(2,1) + DBLE(m2) * dlv(2,2) &
             & + DBLE(m3) * dlv(2,3)
            a3 = DBLE(m1) * dlv(3,1) + DBLE(m2) * dlv(3,2) &
             & + DBLE(m3) * dlv(3,3)

            x1 = rat(1,j) - rat(1,k) - a1
            x2 = rat(2,j) - rat(2,k) - a2
            x3 = rat(3,j) - rat(3,k) - a3

          ELSE

            x1 = rat(1,j) - grot(1,1,m) * rat(1,k) &
             & - grot(1,2,m) * rat(2,k) &
             & - grot(1,3,m) * rat(3,k)

            x2 = rat(2,j) - grot(2,1,m) * rat(1,k) &
             & - grot(2,2,m) * rat(2,k) &
             & - grot(2,3,m) * rat(3,k)

            x3 = rat(3,j) - grot(3,1,m) * rat(1,k) &
             & - grot(3,2,m) * rat(2,k) &
             & - grot(3,3,m) * rat(3,k)

          ENDIF

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)
          xd = 4.0d0 * d

          !=== Born Term For Atom j ===!

          c = sk * rk
          a = d + c
          b = d - c

          IF ( rj < a ) THEN

            a1 = 1.0d0 / a

            IF ( rj > b ) THEN
              a2 = 1.0d0 / rj
            ELSE
              a2 = 1.0d0 / b
            ENDIF

            c = a1 / a2
            c = DLOG(c)

            arg = a * b
            arg = arg * ( a1 + a2 ) - xd
            arg = arg * ( a1 - a2 )
            arg = 0.5d0 * arg + c

            ebr(j) = ebr(j) - arg / xd

          ENDIF

          !=== Born Term For Atom k ===!

          IF ( isym == 1 .and. m /= 0 ) CYCLE

          c = sj * rj
          a = d + c
          b = d - c

          IF ( rk < a ) THEN

            a1 = 1.0d0 / a

            IF ( rk > b ) THEN
              a2 = 1.0d0 / rk
            ELSE
              a2 = 1.0d0 / b
            ENDIF

            c = a1 / a2
            c = DLOG(c)

            arg = a * b
            arg = arg * ( a1 + a2 ) - xd
            arg = arg * ( a1 - a2 )
            arg = 0.5d0 * arg + c

            ebr(k) = ebr(k) - arg / xd

          ENDIF

        ENDDO

        DO i=1,nat

          jtp = indgb(i)
          rj = gbparm(6,jtp)

          ebr(i) = ebr(i) + 1.0d0 / rj
          ebr(i) = 1.0d0 / ebr(i)

          IF ( ebr(i) < rj ) ebr(i) = rj

        ENDDO


        !=== Get Derivatives ===!

        DO i=1,nnpair

          j = lnpair(1,i)
          k = lnpair(2,i)
          m = lnpair(3,i)

          qjk = q(j) * q(k)

          IF ( isym == 0 .or. m == 0 ) THEN

            m2 = MODULO(m,1000000)
            m3 = MODULO(m2,1000)
            m1 = m / 1000000
            m2 = m2 / 1000

            IF ( m1 > 100 ) m1 = 100 - m1
            IF ( m2 > 100 ) m2 = 100 - m2
            IF ( m3 > 100 ) m3 = 100 - m3

            a1 = DBLE(m1) * dlv(1,1) + DBLE(m2) * dlv(1,2) &
             & + DBLE(m3) * dlv(1,3)
            a2 = DBLE(m1) * dlv(2,1) + DBLE(m2) * dlv(2,2) &
             & + DBLE(m3) * dlv(2,3)
            a3 = DBLE(m1) * dlv(3,1) + DBLE(m2) * dlv(3,2) &
             & + DBLE(m3) * dlv(3,3)

            x1 = rat(1,j) - rat(1,k) - a1
            x2 = rat(2,j) - rat(2,k) - a2
            x3 = rat(3,j) - rat(3,k) - a3

          ELSE

            x1 = rat(1,j) - grot(1,1,m) * rat(1,k) &
             & - grot(1,2,m) * rat(2,k) &
             & - grot(1,3,m) * rat(3,k)

            x2 = rat(2,j) - grot(2,1,m) * rat(1,k) &
             & - grot(2,2,m) * rat(2,k) &
             & - grot(2,3,m) * rat(3,k)

            x3 = rat(3,j) - grot(3,1,m) * rat(1,k) &
             & - grot(3,2,m) * rat(2,k) &
             & - grot(3,3,m) * rat(3,k)

          ENDIF

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)

          e = qjk / d

          a = -e / d
          b = 2.0d0 * e
          b = b / xsq

          IF ( ivdw == 1 ) THEN

            req = dvdw(1,j) + dvdw(1,k)
            eps = dvdw(2,j) * dvdw(2,k)
            eps = DSQRT(eps)

            a1 = req / d

            a1 = a1 * a1
            a2 = a1 * a1
            a1 = a1 * a2

            a2 = eps * a1
            a1 = a1 * a2
            a2 = 2.0d0 * a2

            e = e + a1 - a2

            a1 = 12.0d0 * a1
            a2 = 6.0d0 * a2

            a = a + ( a2 - a1 ) / d

            a1 = 13.0d0 * a1
            a2 = 7.0d0 * a2

            b = b + ( a1 - a2 ) / xsq

          ENDIF

          x1 = ebr(j) * ebr(k)
          x2 = 0.25d0 * xsq / x1
          x3 = DEXP(-x2)

          xsq = xsq + x1 * x3
          fgb = DSQRT(xsq)
          qjk = qjk / fgb

          a1 = kappa * fgb
          a2 = a1 * a1
          a3 = DEXP(-a1) / diel

          e = e - qjk + a3 * qjk

          qjk = qjk / xsq
          a3 = a3 * qjk

          a1 = 1.0d0 + a1
          a1 = qjk - a1 * a3
          a2 = a2 * a3 - 3.0d0 * a1

          x1 = 1.0d0 - 0.25d0 * x3
          x2 = 0.50d0 * x2 * x3
          x3 = x1 + x2

          x1 = x1 * d
          x2 = x1 / fgb
          x2 = x2 * x2

          a = a + a1 * x1
          b = b + a2 * x2 + a1 * x3

          IF ( ismth /= 0 .and. d > scut ) THEN

            CALL SMOOTH (d,scut,cut,s,ds,dds)

            x1 = a * s + e * ds
            x2 = 2.0d0 * a * ds
            x2 = b * s + e * dds + x2

            a = x1
            b = x2

          ENDIF

          hnpair(1,i) = hnpair(1,i) + a
          hnpair(2,i) = hnpair(2,i) + b

        ENDDO

        RETURN

      END SUBROUTINE GBORNH
