! ==============================================================================
! Subroutine: SCALEDH
! 
! Purpose: Calculates the correction to the first and second derivatives
!          of the Coulomb and van der Waals potentials between atom
!          pairs required for the calculation of the hessian. The
!          correction takes into account that atoms seperated by three
!          bonds or 1-4 pairs have electrostatic terms scaled by a
!          factor greater than one.
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
!          V(R) = [1/SV - 1] * [A / R^12 - B / R^6] 
!               + [1/SC - 1] * Q'i * Qj / R
!
!          where SV and SC are scaling factors with (SV,SC >= 1)
!          NOTE: Q'i is the scaled charge Qi / SQRT(eps)
!
!          HNPAIR(1,i) = dV/dR
!          HNPAIR(2,i) = d2V/dR2
!
! Arguments:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2009   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE SCALEDH

        USE SystemParam, ONLY : rat,q,dvdw,dlv,sfc,sfv,scut,cut,hnpair,&
                              & lnpair,l14,nnpair,n14,icou,ivdw,ismth

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m,icnt,ipair

        DOUBLE PRECISION :: x1,x2,x3,a1,a2,sc,sv,xsq
        DOUBLE PRECISION :: a,b,d,e,s,ds,dds,req,eps


        IF ( n14 == 0 .or. nnpair == 0 ) RETURN
        IF ( icou == 0 .and. ivdw == 0 ) RETURN


        sc = 1.0d0 / sfc - 1.0d0
        sv = 1.0d0 / sfv - 1.0d0

        DO i=1,n14

          j = l14(1,i)
          k = l14(2,i)

          ipair = 0

          DO m=1,nnpair

            IF ( lnpair(3,m) /= 0 ) CYCLE

            icnt = 0

            IF ( lnpair(1,m) == j ) icnt = icnt + 1
            IF ( lnpair(2,m) == j ) icnt = icnt + 1
            IF ( lnpair(1,m) == k ) icnt = icnt + 1
            IF ( lnpair(2,m) == k ) icnt = icnt + 1

            IF ( icnt == 2 ) THEN

              ipair = m
              EXIT

            ENDIF

          ENDDO

          IF ( ipair == 0 ) CYCLE

          x1 = rat(1,j) - rat(1,k)
          x2 = rat(2,j) - rat(2,k)
          x3 = rat(3,j) - rat(3,k)

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq)

          a = 0.0d0
          b = 0.0d0
          e = 0.0d0

          IF ( icou == 1 ) THEN

            e = q(j) * q(k) / d
            e = sc * e

            a = -e / d
            b = 2.0d0 * e
            b = b / xsq

          ENDIF

          IF ( ivdw == 1 ) THEN

            req = dvdw(1,j) + dvdw(1,k)
            eps = dvdw(2,j) * dvdw(2,k)
            eps = DSQRT(eps)

            a1 = req / d

            a1 = a1 * a1
            a2 = a1 * a1
            a1 = a1 * a2

            a2 = eps * a1
            a1 = a1 * a2
            a2 = 2.0d0 * a2

            a1 = sv * a1
            a2 = sv * a2

            e = e + a1 - a2

            a1 = 12.0d0 * a1
            a2 = 6.0d0 * a2

            a = a + ( a2 - a1 ) / d

            a1 = 13.0d0 * a1
            a2 = 7.0d0 * a2

            b = b + ( a1 - a2 ) / xsq

          ENDIF

          IF ( ismth /= 0 .and. d > scut ) THEN

            CALL SMOOTH (d,scut,cut,s,ds,dds)

            x1 = a * s + e * ds
            x2 = 2.0d0 * a * ds
            x2 = b * s + e * dds + x2

            a = x1
            b = x2

          ENDIF

          hnpair(1,ipair) = hnpair(1,ipair) + a
          hnpair(2,ipair) = hnpair(2,ipair) + b

        ENDDO

        RETURN

      END SUBROUTINE SCALEDH
