! ==============================================================================
! Subroutine: GETBASIS (RAT,DLV,RLV,NAT,RATB,N1,N2,N3)
! 
! Purpose: Calculates the basis vectors for a set of atoms and
!          returns the corresponding integer lattice vector
!          coefficents n1, n2, and n3 such that:
!
!          R_i = B_i + n1 * a1 + n2 * a2 + n3 * a3
!
!          where a1, a2, and a3 are the primitive lattice vectors
!          and R_i are the current atomic coordinates.
!
! Method:
!
! Arguments:
!
!           RAT - Array of dimension (3,NAT) containing the atomic
!                 coordinates.
!           DLV - Array of dimension (3,3) containing the direct
!                 lattice vectors in the columns.
!           RLV - Array of dimension (3,3) containing the reciprocial
!                 lattice vectors in the rows.
!           NAT - Number of atoms
!          RATB - Array of dimension (3,NAT) containing the basis
!                 vectors of the atoms.
!            N1 - Array of dimension (NAT) containing the lattice
!                 coefficients for 1st primitive vector.
!            N2 - Array of dimension (NAT) containing the lattice
!                 coefficients for 2nd primitive vector.
!            N3 - Array of dimension (NAT) containing the lattice
!                 coefficients for 3rd primitive vector.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules -
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE GETBASIS (RAT,DLV,RLV,NAT,RATB,N1,N2,N3)

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        INTEGER, INTENT(IN) :: nat
        INTEGER, INTENT(OUT) :: n1(nat),n2(nat),n3(nat)

        DOUBLE PRECISION, INTENT(IN) :: rat(3,nat),dlv(3,3),rlv(3,3)
        DOUBLE PRECISION, INTENT(OUT) :: ratb(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j1,j2,j3

        DOUBLE PRECISION :: x1,x2,x3


        DO i=1,nat

          x1 = rlv(1,1) * rat(1,i) + rlv(1,2) * rat(2,i) &
           & + rlv(1,3) * rat(3,i)
          x2 = rlv(2,1) * rat(1,i) + rlv(2,2) * rat(2,i) &
           & + rlv(2,3) * rat(3,i)
          x3 = rlv(3,1) * rat(1,i) + rlv(3,2) * rat(2,i) &
           & + rlv(3,3) * rat(3,i)

          j1 = INT(x1)
          j2 = INT(x2)
          j3 = INT(x3)

          IF ( x1 < 0 ) j1 = j1 - 1
          IF ( x2 < 0 ) j2 = j2 - 1
          IF ( x3 < 0 ) j3 = j3 - 1

          x1 = DBLE(j1)
          x2 = DBLE(j2)
          x3 = DBLE(j3)

          n1(i) = j1
          n2(i) = j2
          n3(i) = j3

          ratb(1,i) = rat(1,i) - x1 * dlv(1,1) - x2 * dlv(1,2) &
                  & - x3 * dlv(1,3)
          ratb(2,i) = rat(2,i) - x1 * dlv(2,1) - x2 * dlv(2,2) &
                  & - x3 * dlv(2,3)
          ratb(3,i) = rat(3,i) - x1 * dlv(3,1) - x2 * dlv(3,2) &
                  & - x3 * dlv(3,3)

        ENDDO

        RETURN

      END SUBROUTINE GETBASIS
