! ==============================================================================
! Subroutine: NEIGHBORMAP
! 
! Purpose: Calculates a list of atom pairs that are seperated by
!          less then the cutoff distance.
!
! Method: Allen, Tidesley, Computer Simulation of Liquids
!         Oxford University Press, New York, 1987
!
! Arguments:
!
! Special Notes:
!
!            BXGRID - Length of the edges of a grid spacing used to
!                     construct a set of boxes containing a subset of
!                     the atoms in the full unit cell.
!            NNPAIR - Number of neighbor pairs.
!            LNPAIR - Array of dimension (3,NNPAIR) containing the
!                     list of neighbor pairs.
!
!            LNPAIR(1,i) - The first atom number of the ith pair.
!            LNPAIR(2,i) - The second atom number of the ith pair.
!            LNPAIR(3,i) - The lattice vector "code" for the ith pair.
!
!            The Lattice Vector "code" stores the coefficents for each
!            of the lattice vectors using the following scheme:
!
!            lattice vector code = 123 123 123 (nine digit number)
!                                = ---|---|---
!                                = LV1 LV2 LV3  coefficents for each LV
!
!            The first digit of the number (123) gives the sign
!            1 = negative (or) 0 = positive
!            The remaining two digits of the number (123) give the
!            absolute value of the lattive vector coeeficient = [0-99].
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - GETBASIS, GETMYJOBS
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE NEIGHBORMAP

        USE SystemParam, ONLY : rat,dlv,rlv,cut,vol,bxgrd,nat,nnpair,&
                              & lnpair,ipbc,mpijob,myproc,nproc

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        !=== VARIABLES ===!

        INTEGER :: i,j,k,n,m,i1,i2,i3,j1,j2,j3,k1,k2,k3
        INTEGER :: n1,n2,n3,nb1,nb2,nb3,nm1,nm2,nm3,ntot
        INTEGER :: m1(nat),m2(nat),m3(nat),lbox(nat)
        INTEGER :: ibox(3,nat),istep,icnt,is,ie,ipair

        DOUBLE PRECISION :: ratb(3,nat),rmin(3),rmax(3)
        DOUBLE PRECISION :: xsq,arg,d,x1,x2,x3,t1,t2,t3

        INTEGER, DIMENSION(:,:,:), ALLOCATABLE :: indx


        !=== Check Allocations ===!

        IF ( ALLOCATED(lnpair) ) DEALLOCATE(lnpair)


        !=== If No Periodic Boundary - Make One ===!

        IF ( ipbc == 0 ) THEN

          dlv = 0.0d0
          rlv = 0.0d0

          DO i=1,3

            rmax(i) = rat(i,1)
            rmin(i) = rat(i,1)

          ENDDO

          DO i=1,nat

            rmax(1) = MAX(rat(1,i),rmax(1))
            rmin(1) = MIN(rat(1,i),rmin(1))
            rmax(2) = MAX(rat(2,i),rmax(2))
            rmin(2) = MIN(rat(2,i),rmin(2))
            rmax(3) = MAX(rat(3,i),rmax(3))
            rmin(3) = MIN(rat(3,i),rmin(3))

          ENDDO

          DO i=1,3

            d = rmax(i) - rmin(i)
            d = DABS(d) + 1.0d0

            dlv(i,i) = d
            rlv(i,i) = 1.0d0 / d

          ENDDO

          vol = dlv(1,1) * dlv(2,2) * dlv(3,3)

        ENDIF

        !=== Get Atomic Basis Vectors ===!

        CALL GETBASIS (rat,dlv,rlv,nat,ratb,m1,m2,m3)


        !=== Construct Grid of Boxes ===!

        x1 = dlv(1,1) * dlv(1,1) + dlv(2,1) * dlv(2,1) &
         & + dlv(3,1) * dlv(3,1)
        x2 = dlv(1,2) * dlv(1,2) + dlv(2,2) * dlv(2,2) &
         & + dlv(3,2) * dlv(3,2)
        x3 = dlv(1,3) * dlv(1,3) + dlv(2,3) * dlv(2,3) &
         & + dlv(3,3) * dlv(3,3)

        x1 = DSQRT(x1)
        x2 = DSQRT(x2)
        x3 = DSQRT(x3)

        nb1 = INT( x1 / bxgrd )
        IF ( nb1 == 0 ) nb1 = 1

        nb2 = INT( x2 / bxgrd )
        IF ( nb2 == 0 ) nb2 = 1

        nb3 = INT( x3 / bxgrd )
        IF ( nb3 == 0 ) nb3 = 1

        arg = x1 * x2 * x3
        arg = cut * arg / vol

        x1 = x1 / DBLE(nb1)
        x2 = x2 / DBLE(nb2)
        x3 = x3 / DBLE(nb3)

        nm1 = INT( arg / x1 ) + 1
        nm2 = INT( arg / x2 ) + 1
        nm3 = INT( arg / x3 ) + 1

        ntot = 2 * nm1 + 1
        ntot = ntot * ( 2 * nm2 + 1 )
        ntot = ntot * ( 2 * nm3 + 1 )


        !=== Allocate Work Array ===!

        ALLOCATE ( indx(nb1,nb2,nb3) )

        !=== Form Linked List of Atoms in Each Box ===!

        indx = 0

        DO i=1,nat

          x1 = rlv(1,1) * ratb(1,i) + rlv(1,2) * ratb(2,i) &
           & + rlv(1,3) * ratb(3,i)

          x2 = rlv(2,1) * ratb(1,i) + rlv(2,2) * ratb(2,i) &
           & + rlv(2,3) * ratb(3,i)

          x3 = rlv(3,1) * ratb(1,i) + rlv(3,2) * ratb(2,i) &
           & + rlv(3,3) * ratb(3,i)

          !=== Which box is this atom in ===!

          x1 = x1 * DBLE(nb1)
          x2 = x2 * DBLE(nb2)
          x3 = x3 * DBLE(nb3)

          n1 = INT(x1) + 1
          n2 = INT(x2) + 1
          n3 = INT(x3) + 1

          IF ( n1 > nb1 ) n1 = 1
          IF ( n2 > nb2 ) n2 = 1
          IF ( n3 > nb3 ) n3 = 1

          ibox(1,i) = n1
          ibox(2,i) = n2
          ibox(3,i) = n3

          lbox(i) = indx(n1,n2,n3)
          indx(n1,n2,n3) = i

        ENDDO


        !=== Calculate Neigbormap ===!

        !In 1D Boxes look like this (using 4 boxes per cell)!
        !    Neigh. Cell    !   Main Cell   !  Neig. Cell   !
        !| -3 | -2 | -1 | 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 |
        !Box numbers for each atom are in terms of the M Cell!

        !=== STEP 1 - CALCULATE NUMBER OF NEIGHBORS ===!
        !=== STEP 2 - FILL IN NEIGHBORMAP ===!

        istep = 1

 1      icnt = 0

        DO i=1,nat
        DO n=1,ntot

          n1 = 2 * nm1 + 1
          n2 = 2 * nm2 + 1

          j = n - 1
          k = n1 * n2

          n3 = j / k
          j = j - n3 * k

          n2 = j / n1
          n1 = j - n2 * n1

          j1 = ibox(1,i) + n1 - nm1
          j2 = ibox(2,i) + n2 - nm2
          j3 = ibox(3,i) + n3 - nm3

          k1 = INT( DBLE(j1) / DBLE(nb1) )
          k2 = INT( DBLE(j2) / DBLE(nb2) )
          k3 = INT( DBLE(j3) / DBLE(nb3) )

          IF ( j1 <= 0 .or. MODULO(j1,nb1) == 0 ) k1 = k1 - 1
          IF ( j2 <= 0 .or. MODULO(j2,nb2) == 0 ) k2 = k2 - 1
          IF ( j3 <= 0 .or. MODULO(j3,nb3) == 0 ) k3 = k3 - 1

          j1 = j1 - k1 * nb1
          j2 = j2 - k2 * nb2
          j3 = j3 - k3 * nb3

          !=== Get lattice vector ===!

          k = IABS(k1) + IABS(k2) + IABS(k3)

          t1 = DBLE(k1) * dlv(1,1) + DBLE(k2) * dlv(1,2) &
           & + DBLE(k3) * dlv(1,3)
          t2 = DBLE(k1) * dlv(2,1) + DBLE(k2) * dlv(2,2) &
           & + DBLE(k3) * dlv(2,3)
          t3 = DBLE(k1) * dlv(3,1) + DBLE(k2) * dlv(3,2) &
           & + DBLE(k3) * dlv(3,3)

          !=== Check neighbors ===!

          j = indx(j1,j2,j3)

          DO WHILE ( j /= 0 .and. j >= i )

            x1 = ratb(1,i) - ratb(1,j) - t1
            x2 = ratb(2,i) - ratb(2,j) - t2
            x3 = ratb(3,i) - ratb(3,j) - t3

            xsq = x1 * x1 + x2 * x2 + x3 * x3
            d = DSQRT(xsq)

            IF ( d <= cut ) THEN
            IF ( i /= j .or. k /= 0 ) THEN

              i1 = k1 + m1(i) - m1(j)
              i2 = k2 + m2(i) - m2(j)
              i3 = k3 + m3(i) - m3(j)

              IF ( i1 < 0 ) i1 = 100 + IABS(i1)
              IF ( i2 < 0 ) i2 = 100 + IABS(i2)
              IF ( i3 < 0 ) i3 = 100 + IABS(i3)

              m = i1 * 1000000 + i2 * 1000 + i3

              !=== If No PBC --> Only Except if m = 0 ===!

              IF ( ipbc /= 0 .or. m == 0 ) THEN

                icnt = icnt + 1

                IF ( istep == 2 ) THEN
                IF ( icnt >= is .and. icnt <= ie ) THEN

                  ipair = icnt - is + 1

                  lnpair(1,ipair) = i
                  lnpair(2,ipair) = j
                  lnpair(3,ipair) = m

                ENDIF
                ENDIF

              ENDIF

            ENDIF
            ENDIF

            j = lbox(j)

          ENDDO

        ENDDO
        ENDDO


        !=== Allocate Neighbor Pair List ===!

        IF ( istep == 1 ) THEN

          CALL GETMYJOBS (is,ie,icnt,mpijob,myproc,nproc)

          nnpair = ie - is + 1

          ALLOCATE (lnpair(3,nnpair))

          istep = 2

          GOTO 1

        ENDIF


        !=== If No PBC --> Reset Lattice Vectors ===!

        IF ( ipbc == 0 ) THEN

          dlv = 0.0d0
          rlv = 0.0d0
          vol = 0.0d0

        ENDIF

        !=== Deallocate Work Arrays ===!

        DEALLOCATE (indx)

        RETURN

      END SUBROUTINE NEIGHBORMAP
