! ==============================================================================
! Subroutine: SYMMETRYMAP
! 
! Purpose: Calculates a list of atom pairs that are seperated by
!          less then the cutoff distance. This is used for systems
!          that have rotational symmetry operators that generate
!          the remaning atoms in the system from a primitive set.
!
! Method: Allen, Tidesley, Computer Simulation of Liquids
!         Oxford University Press, New York, 1987
!
! Arguments:
!
! Special Notes:
!
!            BXGRID - Length of the edges of a grid spacing used to
!                     construct a set of boxes containing a subset of
!                     the atoms in the full unit cell.
!            MGPRM  - Array of dimension (NGRP) which stores a map
!                     of permutations for the primitive cell.
!                     GROT(:,:,iGRP) takes the prim. cell (site 1)
!                     to site number MGPRM(iGRP).
!            NNPAIR - Number of neighbor pairs.
!            LNPAIR - Array of dimension (3,NNPAIR) containing the list
!                     of neighbor pairs.
!
!            LNPAIR(1,i) = The first atom number of the ith pair.
!            LNPAIR(2,i) = The second atom number of the ith pair.
!            LNPAIR(3,i) = The group rotation operator for the ith pair.
!
!            LNPAIR(3,i) stores the number [0,NGRP] of the group
!            rotation operator that must be applied to the second
!            atom of the pair in order to produce a neighbor pair.
!            IF LNPAIR(3,i) = 0 then the group rotation operator
!            is simply the 3 X 3 identity matrix.
!
!            ------------- NOTE ON NEIGHBOR INTERACTIONS ---------------
!            For group symmetry situations, if atom i at the primitive
!            site is neighbors with atom j at another site, the pair
!            interaction will only contribute a force to atom i at the
!            primitive site. This is because the force on atom j at the
!            primitive site is not the negative of the force but is instead
!            related by a rotation operation. Thus i,j will BOTH range
!            from 1,NAT if atom j is at another site.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - GETMYJOBS
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE SYMMETRYMAP

        USE SystemParam, ONLY : rat,grot,cut,bxgrd,nat,nnpair,lnpair,&
                              & mgprm,ngrp,nsite,mpijob,myproc,nproc

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        !=== VARIABLES ===!

        INTEGER :: i,j,m,isite,istep,n1,n2,n3,lbox(nat)
        INTEGER :: nb1,nb2,nb3,nm1,nm2,nm3,lsite(ngrp)
        INTEGER :: ns1,ns2,ns3,ne1,ne2,ne3,icnt,is,ie,ipair

        DOUBLE PRECISION :: x1,x2,x3,bx1,bx2,bx3,rc1,rc2,rc3
        DOUBLE PRECISION :: rmin(3),rmax(3),xsq,arg,d

        INTEGER, DIMENSION(:,:,:), ALLOCATABLE :: indx


        !=== Check Allocations ===!

        IF ( ALLOCATED(lnpair) ) DEALLOCATE(lnpair)


        !=== Find Dimensions of Bounding Box ===!

        DO i=1,3

          rmax(i) = rat(i,1)
          rmin(i) = rat(i,1)

        ENDDO

        DO i=1,nat

          rmax(1) = MAX(rat(1,i),rmax(1))
          rmin(1) = MIN(rat(1,i),rmin(1))
          rmax(2) = MAX(rat(2,i),rmax(2))
          rmin(2) = MIN(rat(2,i),rmin(2))
          rmax(3) = MAX(rat(3,i),rmax(3))
          rmin(3) = MIN(rat(3,i),rmin(3))

        ENDDO

        bx1 = DABS( rmax(1) - rmin(1) ) + 1.0d0
        bx2 = DABS( rmax(2) - rmin(2) ) + 1.0d0
        bx3 = DABS( rmax(3) - rmin(3) ) + 1.0d0

        rc1 = 0.50d0 * ( rmax(1) + rmin(1) )
        rc2 = 0.50d0 * ( rmax(2) + rmin(2) )
        rc3 = 0.50d0 * ( rmax(3) + rmin(3) )

        nb1 = INT( bx1 / bxgrd )
        IF ( nb1 == 0 ) nb1 = 1

        nb2 = INT( bx2 / bxgrd )
        IF ( nb2 == 0 ) nb2 = 1

        nb3 = INT( bx3 / bxgrd )
        IF ( nb3 == 0 ) nb3 = 1


        !=== Find the sites that are potential ===!
        !=== neighbors of first cell ===!

        arg = bx1 * bx1 + bx2 * bx2 + bx3 * bx3
        arg = DSQRT(arg)

        lsite(1) = 1

        DO isite=2,nsite

          lsite(isite) = 0

          !=== Find group rotation that ===!
          !=== takes Site 1 --> Site iS ===!

          DO i=1,ngrp
          IF ( mgprm(i) == isite ) m = i
          ENDDO

          !=== Get distance between sphere centers ===!

          x1 = grot(1,1,m) * rc1 + grot(1,2,m) * rc2 &
           & + grot(1,3,m) * rc3 - rc1

          x2 = grot(2,1,m) * rc1 + grot(2,2,m) * rc2 &
           & + grot(2,3,m) * rc3 - rc2

          x3 = grot(3,1,m) * rc1 + grot(3,2,m) * rc2 &
           & + grot(3,3,m) * rc3 - rc3 

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          d = DSQRT(xsq) - cut

          !=== If d < 2 * (Sphere Radius) then this ===!
          !=== site may have neighbors with site 1 ===!

          IF ( d <= arg ) lsite(isite) = 1

        ENDDO


        !=== Allocate Work Array ===!

        ALLOCATE ( indx(nb1,nb2,nb3) )

        !=== Form Linked List of Atoms in Each Box ===!

        indx = 0

        DO i=1,nat

          x1 = ( rat(1,i) - rmin(1) ) / bx1
          x2 = ( rat(2,i) - rmin(2) ) / bx2
          x3 = ( rat(3,i) - rmin(3) ) / bx3

          !=== Which box is this atom in ===!

          x1 = x1 * DBLE(nb1)
          x2 = x2 * DBLE(nb2)
          x3 = x3 * DBLE(nb3)

          n1 = INT(x1) + 1
          n2 = INT(x2) + 1
          n3 = INT(x3) + 1

          IF ( n1 > nb1 ) n1 = nb1
          IF ( n2 > nb2 ) n2 = nb2
          IF ( n3 > nb3 ) n3 = nb3

          lbox(i) = indx(n1,n2,n3)
          indx(n1,n2,n3) = i

        ENDDO


        !=== Construct Grid of Boxes ===!

        bx1 = bx1 / DBLE(nb1)
        bx2 = bx2 / DBLE(nb2)
        bx3 = bx3 / DBLE(nb3)

        nm1 = INT( cut / bx1 ) + 1
        nm2 = INT( cut / bx2 ) + 1
        nm3 = INT( cut / bx3 ) + 1


        !=== Calculate Neigbormap ===!

        !In 1D Boxes look like this (using 4 boxes per cell)!
        !    Neigh. Cell    !   Main Cell   !  Neig. Cell   !
        !| -3 | -2 | -1 | 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 |
        !Box numbers for each atom are in terms of the M Cell!

        !=== STEP 1 - CALCULATE NUMBER OF NEIGHBORS ===!
        !=== STEP 2 - FILL IN NEIGHBORMAP ===!

        istep = 1

 1      icnt = 0

        DO i=1,nat
        DO isite=1,nsite
        IF ( lsite(isite) == 1 ) THEN

          DO j=1,ngrp
          IF ( mgprm(j) == isite ) m = j
          ENDDO

          !=== Rotate atom i by R^-1 = R^T ===!
          !=== We want to find the box that ===!
          !=== atom i is in relative to site is ===!

          x1 = grot(1,1,m) * rat(1,i) + grot(2,1,m) * rat(2,i) &
           & + grot(3,1,m) * rat(3,i) - rmin(1)

          x2 = grot(1,2,m) * rat(1,i) + grot(2,2,m) * rat(2,i) &
           & + grot(3,2,m) * rat(3,i) - rmin(2)

          x3 = grot(1,3,m) * rat(1,i) + grot(2,3,m) * rat(2,i) &
           & + grot(3,3,m) * rat(3,i) - rmin(3)

          x1 = x1 / bx1
          x2 = x2 / bx2
          x3 = x3 / bx3

          IF ( x1 < 0.0d0 ) x1 = x1 - 1.0d0
          IF ( x2 < 0.0d0 ) x2 = x2 - 1.0d0
          IF ( x3 < 0.0d0 ) x3 = x3 - 1.0d0

          ns1 = INT(x1) + 1
          ne1 = ns1 + nm1
          ns1 = ns1 - nm1
          ns1 = MAX(ns1,1)
          ne1 = MAX(ne1,1)
          ns1 = MIN(ns1,nb1)
          ne1 = MIN(ne1,nb1)

          ns2 = INT(x2) + 1
          ne2 = ns2 + nm2
          ns2 = ns2 - nm2
          ns2 = MAX(ns2,1)
          ne2 = MAX(ne2,1)
          ns2 = MIN(ns2,nb2)
          ne2 = MIN(ne2,nb2)

          ns3 = INT(x3) + 1
          ne3 = ns3 + nm3
          ns3 = ns3 - nm3
          ns3 = MAX(ns3,1)
          ne3 = MAX(ne3,1)
          ns3 = MIN(ns3,nb3)
          ne3 = MIN(ne3,nb3)

          DO n3=ns3,ne3
          DO n2=ns2,ne2
          DO n1=ns1,ne1

            j = indx(n1,n2,n3)

            DO WHILE ( j /= 0 )

              x1 = grot(1,1,m) * rat(1,j) + grot(1,2,m) * rat(2,j) &
               & + grot(1,3,m) * rat(3,j) - rat(1,i)

              x2 = grot(2,1,m) * rat(1,j) + grot(2,2,m) * rat(2,j) &
               & + grot(2,3,m) * rat(3,j) - rat(2,i)

              x3 = grot(3,1,m) * rat(1,j) + grot(3,2,m) * rat(2,j) &
               & + grot(3,3,m) * rat(3,j) - rat(3,i)

              xsq = x1 * x1 + x2 * x2 + x3 * x3
              d = DSQRT(xsq)

              IF ( d <= cut ) THEN
              IF ( isite /= 1 .or. j > i ) THEN

                icnt = icnt + 1

                IF ( istep == 2 ) THEN
                IF ( icnt >= is .and. icnt <= ie ) THEN

                  ipair = icnt - is + 1

                  lnpair(1,ipair) = i
                  lnpair(2,ipair) = j
                  lnpair(3,ipair) = 0

                  IF ( isite /= 1 ) lnpair(3,ipair) = m

                ENDIF
                ENDIF

              ENDIF
              ENDIF

              j = lbox(j)

            ENDDO

          ENDDO
          ENDDO
          ENDDO

        ENDIF
        ENDDO
        ENDDO


        !=== Allocate Neighbor Pair List ===!

        IF ( istep == 1 ) THEN

          CALL GETMYJOBS (is,ie,icnt,mpijob,myproc,nproc)

          nnpair = ie - is + 1

          ALLOCATE (lnpair(3,nnpair))

          istep = 2

          GOTO 1

        ENDIF


        !=== Deallocate Work Arrays ===!

        DEALLOCATE (indx)

        RETURN

      END SUBROUTINE SYMMETRYMAP
