! ==============================================================================
! Subroutine: BSPLINE (X,A,AP,N)
! 
! Purpose: Calculates the order N cardinal B-spline of fractional argument
!          x = [0,1] and the derivatives at the points u = (x + i - 1) i=[1,n].
!
! Method: Recursive algorithm using the identities:
!
!         Mn(u) = [u * Mn-1(u) + (n-u) * Mn-1(u-1)] / (n-1)
!
!         d(Mn)/du = Mn-1(u) - Mn-1(u-1)
!
! Arguments:
!
!           X  - Real fractional argument which ranges from [0,1].
!           A  - Real array of length N. On OUTPUT, contains the values
!                A(i) = Mn(u) where u = (x + i - 1).
!           AP - Real array of length N. On OUTPUT, contains the values
!                AP(i) = dMn(u)/du where u = (x + i - 1).
!           N  - Order of the Cardinal B-spline and dimensions of
!                the arrays A and AP.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules -
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE BSPLINE (X,A,AP,N)

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        INTEGER, INTENT(IN) :: n

        DOUBLE PRECISION, INTENT(IN) :: x
        DOUBLE PRECISION, INTENT(OUT) :: a(n),ap(n)

        !=== VARIABLES ===!

        INTEGER :: i,iord

        DOUBLE PRECISION :: c,u1,u2


        !=== Initalize ===!

        a = 0.0d0
        ap = 0.0d0


        !=== Second order spline ===!

        a(1) = x
        a(2) = 1.0d0 - x

        ap(1) = 1.0d0
        ap(2) = -1.0d0

        !=== Recursion ===!

        iord = 2

        DO WHILE ( iord < n )

          c = 1.0d0 / DBLE(iord)

          iord = iord + 1

          IF ( iord == n ) THEN

            ap(1) = a(1)

            DO i=2,iord

              ap(i) = a(i) - a(i-1)

            ENDDO

          ENDIF

          u1 = x + DBLE(iord-1)
          u2 = 1.0d0 - x

          DO i=iord,2,-1

            a(i) = u1 * a(i) + u2 * a(i-1)

            a(i) = c * a(i)

            u1 = u1 - 1.0d0
            u2 = u2 + 1.0d0

          ENDDO

          a(1) = c * x * a(1)

        ENDDO

        RETURN

      END SUBROUTINE BSPLINE
