! ==============================================================================
! Subroutine: LINEMIN (P)
! 
! Purpose: Performs a line minimization along a serch direction.
!
! Method: Finds the root of the one dimensional functional G = <F|P> where
!         F is the force and P is the search direction. A linear approximation
!         of the step size is used as an intial guess followed by the false
!         position method (Newton-Rhapson) to improve the step size until
!         convergange.
!
! Arguments:
!
!           P - Array of dimension (3,NAT)
!               Contains the search direction.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - ENERGY
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE LINEMIN (P)

        USE SystemParam, ONLY : rat,nat

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(IN) :: p(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,iit,iflg

        DOUBLE PRECISION :: xa,xb,xc,ga,gb,gc
        DOUBLE PRECISION :: eb,ea,ed,ec,ev,pm,fm
        DOUBLE PRECISION :: f(3,nat),rold(3,nat)

        INTEGER, PARAMETER :: maxit = 50

        DOUBLE PRECISION, PARAMETER :: tol = 1.0d-10
        DOUBLE PRECISION, PARAMETER :: sml = 1.0d-5
        DOUBLE PRECISION, PARAMETER :: gmean = 1.61803398874989490252d0


        iflg = 0

        !=== Inital Energy Call ===!

        CALL ENERGY (eb,ea,ed,ec,ev,f)


        !=== Find Intial G = <F|P> ===!

        xa = 0.0d0
        ga = 0.0d0
        pm = 0.0d0
        fm = 0.0d0

        DO i=1,nat

          ga = ga + p(1,i) * f(1,i)
          ga = ga + p(2,i) * f(2,i)
          ga = ga + p(3,i) * f(3,i)

          fm = fm + f(1,i) * f(1,i)
          fm = fm + f(2,i) * f(2,i)
          fm = fm + f(3,i) * f(3,i)

          pm = pm + p(1,i) * p(1,i)
          pm = pm + p(2,i) * p(2,i)
          pm = pm + p(3,i) * p(3,i)

          rold(1,i) = rat(1,i)
          rold(2,i) = rat(2,i)
          rold(3,i) = rat(3,i)

        ENDDO

        fm = 1.0d0 / DSQRT(fm)
        pm = 1.0d0 / DSQRT(pm)

        ga = ga * fm * pm


        !=== Guess Inital Step ===!

        xb = sml

        DO i=1,nat

          rat(1,i) = rold(1,i) + xb * p(1,i)
          rat(2,i) = rold(2,i) + xb * p(2,i)
          rat(3,i) = rold(3,i) + xb * p(3,i)

        ENDDO

        CALL ENERGY (eb,ea,ed,ec,ev,f)

        gb = 0.0d0

        DO i=1,nat

          gb = gb + p(1,i) * f(1,i)
          gb = gb + p(2,i) * f(2,i)
          gb = gb + p(3,i) * f(3,i)

        ENDDO

        gb = gb * fm * pm

        !=== Linear Approximation ===!

        xb = -ga * xb
        xb = xb / ( gb - ga )

        DO i=1,nat

          rat(1,i) = rold(1,i) + xb * p(1,i)
          rat(2,i) = rold(2,i) + xb * p(2,i)
          rat(3,i) = rold(3,i) + xb * p(3,i)

        ENDDO

        CALL ENERGY (eb,ea,ed,ec,ev,f)

        gb = 0.0d0

        DO i=1,nat

          gb = gb + p(1,i) * f(1,i)
          gb = gb + p(2,i) * f(2,i)
          gb = gb + p(3,i) * f(3,i)

        ENDDO

        gb = gb * fm * pm


        !=== Bound the Zero ===!

        DO WHILE ( ga * gb > 0.0d0 )

          ga = gb
          xa = xb

          xb = xb * gmean

          DO i=1,nat

            rat(1,i) = rold(1,i) + xb * p(1,i)
            rat(2,i) = rold(2,i) + xb * p(2,i)
            rat(3,i) = rold(3,i) + xb * p(3,i)

          ENDDO

          CALL ENERGY (eb,ea,ed,ec,ev,f)

          gb = 0.0d0

          DO i=1,nat

            gb = gb + p(1,i) * f(1,i)
            gb = gb + p(2,i) * f(2,i)
            gb = gb + p(3,i) * f(3,i)

          ENDDO

          gb = gb * fm * pm

        ENDDO


        !=== Locate root of <F|P> ===!

        DO iit=1,maxit

          xc = ( gb * xa - ga * xb ) / ( gb - ga )

          DO i=1,nat

            rat(1,i) = rold(1,i) + xc * p(1,i)
            rat(2,i) = rold(2,i) + xc * p(2,i)
            rat(3,i) = rold(3,i) + xc * p(3,i)

          ENDDO

          CALL ENERGY (eb,ea,ed,ec,ev,f)

          gc = 0.0d0

          DO i=1,nat

            gc = gc + p(1,i) * f(1,i)
            gc = gc + p(2,i) * f(2,i)
            gc = gc + p(3,i) * f(3,i)

          ENDDO

          gc = gc * fm * pm

          IF ( gc * gb > 0.0d0 ) THEN

            xb = xc
            gb = gc

            IF ( iflg == 2 ) ga = 0.50d0 * ga

            iflg = 2

          ELSEIF ( gc * ga > 0.0d0 ) THEN

            xa = xc
            ga = gc

            IF ( iflg == 1 ) gb = 0.50d0 * gb

            iflg = 1

          ELSE

            EXIT

          ENDIF

          IF ( DABS(xa-xb) < tol ) EXIT

        ENDDO

        RETURN

      END SUBROUTINE LINEMIN
