! ==============================================================================
! Subroutine: MATINV (A)
! 
! Purpose: Computes the inverse of a 3 X 3 matrix.
!
! Method: Calculates the inverse using the cross product method.
!
! Arguments:
!
!           A - Array of dimensions A(3,3).
!               ON INPUT - contains the 3 X 3 matrix to be inverted.
!               ON OUTPUT- contains the inverse of the 3 X 3 matrix.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules -
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE MATINV (A)

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(INOUT) :: a(3,3)

        !=== VARIABLES ===!

        DOUBLE PRECISION :: x1,x2,x3,y1,y2,y3
        DOUBLE PRECISION :: z1,z2,z3,det

        DOUBLE PRECISION, PARAMETER :: sml = 1.0d-6


        x1 = a(2,2) * a(3,3) - a(3,2) * a(2,3)
        x2 = a(3,2) * a(1,3) - a(1,2) * a(3,3)
        x3 = a(1,2) * a(2,3) - a(2,2) * a(1,3)

        y1 = a(2,3) * a(3,1) - a(3,3) * a(2,1)
        y2 = a(3,3) * a(1,1) - a(1,3) * a(3,1)
        y3 = a(1,3) * a(2,1) - a(2,3) * a(1,1)

        z1 = a(2,1) * a(3,2) - a(3,1) * a(2,2)
        z2 = a(3,1) * a(1,2) - a(1,1) * a(3,2)
        z3 = a(1,1) * a(2,2) - a(2,1) * a(1,2)

        det = x1 * a(1,1) + x2 * a(2,1) + x3 * a(3,1)

        IF ( DABS(det) < sml ) THEN

          WRITE(4,*)'ERROR: MATINV could not form inverse of matrix.'

          STOP

        ENDIF

        a(1,1) = x1 / det
        a(2,1) = x2 / det
        a(3,1) = x3 / det

        a(1,2) = y1 / det
        a(2,2) = y2 / det
        a(3,2) = y3 / det

        a(1,3) = z1 / det
        a(2,3) = z2 / det
        a(3,3) = z3 / det

        RETURN

      END SUBROUTINE MATINV
