! ==============================================================================
! Subroutine: MAXWELLDIST (TEMP,AMASS,NAT,ISEED,VEL)
! 
! Purpose: Given an intial temperature, calculates the atomic
!          velocities from a probability denisity which satisfies
!          a Maxwell distribution at temperature T.
!
! Method: Uses a random number generator on the interval [0,1] to choose
!         velocites of each atom according to a Maxwell probability
!         distribution. On average, each atom should have M<V> = 3kT.
!
! Arguments:
!
!           TEMP  - Temperature of the atoms.
!           AMASS - Array of dimension NAT containing the atomic
!                   masses.
!           NAT   - Total number of atoms in the system.
!           ISEED - Seed for random number generator.
!           VEL   - Array of dimension (3,NAT) containing the
!                   velocities of each atom.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules -
! Functions - RANDOM, DIERFC
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE MAXWELLDIST (TEMP,AMASS,NAT,ISEED,VEL)

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        INTEGER, INTENT(IN) :: nat
        INTEGER, INTENT(INOUT) :: iseed

        DOUBLE PRECISION, INTENT(IN) :: temp,amass(nat)
        DOUBLE PRECISION, INTENT(OUT) :: vel(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i

        DOUBLE PRECISION :: a,c,x1,x2,x3,fac
        DOUBLE PRECISION :: RANDOM,DIERFC

        DOUBLE PRECISION, PARAMETER :: cfac = 9.6485339840d3
        DOUBLE PRECISION, PARAMETER :: boltz = 8.61734315d-5


        c = boltz * temp * cfac
        c = 2.0d0 * c

        DO i=1,nat

          a = amass(i)

          fac = c / a
          fac = DSQRT(fac)

          x1 = 2.0d0 * RANDOM(iseed)
          x2 = 2.0d0 * RANDOM(iseed)
          x3 = 2.0d0 * RANDOM(iseed)

          x1 = fac * DIERFC(x1)
          x2 = fac * DIERFC(x2)
          x3 = fac * DIERFC(x3)

          vel(1,i) = x1
          vel(2,i) = x2
          vel(3,i) = x3

        ENDDO

        RETURN

      END SUBROUTINE MAXWELLDIST
