! ==============================================================================
! Subroutine: SMOOTH (R,SCUT,CUT,S,DS,DDS)
! 
! Purpose: Computes the value of the smoothing function and its first
!          and second derivatives. The smoothing function is used to
!          smooth the potential energy V(r) and its first and second
!          derivatives to zero at r = cutoff.
!
! Method: The smoothing function is defined as:
!
!         S(r) = 1 - x^3 * [ 6*x^2 - 15*x + 10 ]
!
!         where [ using R = (cut-scut)/2 ]
!
!         x = 0                      FOR   r <= scut
!         x = ( r - scut ) / R       FOR   scut < r < R+scut
!         x = 1                      FOR   R+scut <= r
!
! Arguments:
!
!           R    - Argument of the smoothing function.
!           SCUT - Value of r at which the smoothing function
!                  turns on.
!           CUT  - Cutoff value used to compute electrostatic
!                  interactions. The smoothing function will
!                  smooth V(r) to zero at r = CUT.
!           S    - Value of the smoothing function at r.
!           DS   - Value of the first derivative of S at r.
!           DDS  - Value of the second derivative of S at r.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules -
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE SMOOTH (R,SCUT,CUT,S,DS,DDS)

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(IN) :: r,scut,cut
        DOUBLE PRECISION, INTENT(OUT) :: s,ds,dds

        !=== VARIABLES ===!

        DOUBLE PRECISION :: x,xsq,arg,x1,x2,c1,c2,rc


        rc = ( cut - scut ) / 2.0d0
        x = rc + scut

        IF ( r <= scut ) THEN

          s = 1.0d0
          ds = 0.0d0
          dds = 0.0d0

          RETURN

        ELSEIF ( x <= r ) THEN

          s = 0.0d0
          ds = 0.0d0
          dds = 0.0d0

          RETURN

        ENDIF


        x = ( r - scut ) / rc
        x1 = x - 1.0d0
        x2 = x - 2.0d0

        xsq = x * x
        arg = x * x1

        c1 = 6.0d0 * arg / rc
        c2 = 10.0d0 * c1 / rc
        arg = 5.0d0 * arg

        s = 1.0d0 - xsq * ( arg * x2 + x * xsq )

        ds = - c1 * arg

        dds = - c2 * ( x1 + x )

        RETURN

      END SUBROUTINE SMOOTH
