! ==============================================================================
! Subroutine: ZEROP (RAT,VEL,AMASS,NAT)
! 
! Purpose: Sets the linear and angular momentum of all atoms about
!          the center of mass to zero.
!
! Method: For the linear momentum:
!
!         Computes P = Sum_i mi * vi and M = Sum_i mi
!         V = P / M is subtracted from the velocites.
!
!         For the angular momentum:
!
!         Computes L = Sum_i mi * (Ri X vi) = Sum_i mi * Ri X (w X Ri)
!         where Ri = ri - R is the distance of atom i from the center
!         of mass R, and w is the angular velocity which can be calculated
!         from L = I * w. V = (w X Ri) is subtracted from the velocites.
!
! Arguments:
!
!           RAT   - Array of dimension (3,NAT) containing the
!                   positions of each atom.
!           VEL   - Array of dimension (3,NAT) containing the
!                   velocities of each atom.
!           AMASS - Array of dimension NAT containing the atomic
!                   masses.
!           NAT   - Total number of atoms in the system.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - MATINV
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE ZEROP (RAT,VEL,AMASS,NAT)

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        INTEGER, INTENT(IN) :: nat

        DOUBLE PRECISION, INTENT(IN) :: rat(3,nat),amass(nat)
        DOUBLE PRECISION, INTENT(INOUT) :: vel(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i

        DOUBLE PRECISION :: x1,x2,x3,v1,v2,v3,t(3,3)
        DOUBLE PRECISION :: rc1,rc2,rc3,vc1,vc2,vc3
        DOUBLE PRECISION :: w1,w2,w3,a,atot


        x1 = 0.0d0
        x2 = 0.0d0
        x3 = 0.0d0

        v1 = 0.0d0
        v2 = 0.0d0
        v3 = 0.0d0

        atot = 0.0d0


        DO i=1,nat

          a = amass(i)

          atot = atot + a

          x1 = x1 + a * rat(1,i)
          x2 = x2 + a * rat(2,i)
          x3 = x3 + a * rat(3,i)

          v1 = v1 + a * vel(1,i)
          v2 = v2 + a * vel(2,i)
          v3 = v3 + a * vel(3,i)

        ENDDO

        rc1 = x1 / atot
        rc2 = x2 / atot
        rc3 = x3 / atot

        vc1 = v1 / atot
        vc2 = v2 / atot
        vc3 = v3 / atot


        v1 = 0.0d0
        v2 = 0.0d0
        v3 = 0.0d0

        DO i=1,nat

          a = amass(i)

          vel(1,i) = vel(1,i) - vc1
          vel(2,i) = vel(2,i) - vc2
          vel(3,i) = vel(3,i) - vc3

          w1 = vel(1,i)
          w2 = vel(2,i)
          w3 = vel(3,i)

          x1 = rat(1,i) - rc1
          x2 = rat(2,i) - rc2
          x3 = rat(3,i) - rc3

          v1 = v1 + a * ( x2 * w3 - x3 * w2 )
          v2 = v2 + a * ( x3 * w1 - x1 * w3 )
          v3 = v3 + a * ( x1 * w2 - x2 * w1 )

          w1 = a * x1
          w2 = a * x2
          w3 = a * x3

          t(1,2) = t(1,2) - w1 * x2
          t(1,3) = t(1,3) - w1 * x3
          t(2,3) = t(2,3) - w2 * x3

          w1 = w1 * x1 
          w2 = w2 * x2 
          w3 = w3 * x3

          t(1,1) = t(1,1) + w2 + w3
          t(2,2) = t(2,2) + w1 + w3
          t(3,3) = t(3,3) + w1 + w2

        ENDDO

        t(2,1) = t(1,2)
        t(3,1) = t(1,3)
        t(3,2) = t(2,3)

        CALL MATINV (t)

        w1 = t(1,1) * v1 + t(1,2) * v2 + t(1,3) * v3
        w2 = t(2,1) * v1 + t(2,2) * v2 + t(2,3) * v3
        w3 = t(3,1) * v1 + t(3,2) * v2 + t(3,3) * v3


        DO i=1,nat

          x1 = rat(1,i) - rc1
          x2 = rat(2,i) - rc2
          x3 = rat(3,i) - rc3

          v1 = w2 * x3 - w3 * x2
          v2 = w3 * x1 - w1 * x3
          v3 = w1 * x2 - w2 * x1

          vel(1,i) = vel(1,i) - v1
          vel(2,i) = vel(2,i) - v2
          vel(3,i) = vel(3,i) - v3

        ENDDO

        RETURN

      END SUBROUTINE ZEROP
