! ==============================================================================
! Subroutine: EWALDSETUP
! 
! Purpose: Computes the number of Reciprocal lattice vectors to sum over
!          such that the remainding terms in the ewald reciprocal sum are
!          less than a specified tolerance.
!
! Method: Uses a recursion to find the reciprocal cutoff to a tolerance of
!         2^-50 ~ 1E-15. Then finds the maximum number of reciprocal vectors
!         required in the ewald reciprocal sum term.
!
! Arguments:
!
!           TOLE - Tolerance for direct and reciprocal Ewald sums.
!           BETA - Ewald beta factor.
!           VOL  - Volume of the real space primitive unit cell.
!           RLV  - Real array of dimension 3 X 3 containing the primitive
!                  reciprocal lattice vectors in the rows.
!                  NOTE: These vestors are NOT multiplied by 2 PI.
!           MREC - Array of length 3. On OUTPUT contains the number of
!                  reciprocal vectors to sum over. The reciprocal triple
!                  sum will range from -MREC to +MREC along each vector.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE EWALDSETUP

        USE SystemParam, ONLY : tole,beta,vol,rlv,mrec

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        !=== VARIABLES ===!

        INTEGER :: i,j

        DOUBLE PRECISION :: c,glo,ghi,gm,gcut
        DOUBLE PRECISION :: arg,g1,g2,g3

        DOUBLE PRECISION, PARAMETER :: pi = 3.14159265358979323846d0


        !=== Bracket Solution ===!

        c = pi / beta
        c = c * c

        i = 1
        ghi = 1.0d0
        arg = ghi * c
        arg = DEXP(-arg) / ghi

        DO WHILE ( arg > tole )

          i = i + 1
          ghi = 2.0d0 * ghi
          arg = ghi * c
          arg = DEXP(-arg) / ghi

        ENDDO

        !=== Find Squared Reciprocal Cutoff ===!

        j = i + 50
        glo = 0.0d0

        DO i=1,j

          gm = 0.50d0 * ( glo + ghi )
          arg = gm * c
          arg = DEXP(-arg) / gm

          IF ( arg > tole ) THEN

            glo = gm

          ELSE

            ghi = gm

          ENDIF

        ENDDO

        gcut = DSQRT(gm)


        !=== Compute Lengths of Rec Vectors ===!

        g1 = rlv(1,1) * rlv(1,1) + rlv(1,2) * rlv(1,2) &
         & + rlv(1,3) * rlv(1,3)
        g2 = rlv(2,1) * rlv(2,1) + rlv(2,2) * rlv(2,2) &
         & + rlv(2,3) * rlv(2,3)
        g3 = rlv(3,1) * rlv(3,1) + rlv(3,2) * rlv(3,2) &
         & + rlv(3,3) * rlv(3,3)

        g1 = DSQRT(g1)
        g2 = DSQRT(g2)
        g3 = DSQRT(g3)

        arg = gcut * vol

        gm = arg * g2 * g3
        mrec(1) = INT(gm)

        gm = arg * g1 * g3
        mrec(2) = INT(gm)

        gm = arg * g1 * g2
        mrec(3) = INT(gm)

        RETURN

      END SUBROUTINE EWALDSETUP
