! ==============================================================================
! Subroutine: SETUPGB
! 
! Purpose: Sets up the parameters for the SASA ans generalized Born models.
!
! Method:   SASA Parameters:
!          (1) W.C. Still et al., J. Comp. Chem., 20 217 (1999).
!          (2) J. Phys. Chem., 98 1978-1988 (1994).
!
!           Generalized Born Parameters:
!          (1) Hawkins et al., J. Phys. Chem., 100 19824 (1996).
!          (2) V. Tsui and D.A. Case, Biopolymers, 56 275 (2000).
!
! Arguments:
!
!           GBPARM - Array of dimension (7,NGBP) containing the list of
!                    parameters needed for both the SASA and generalized
!                    Born model calculations.
!                    (1,i) = SASA RADIUS for atom type i
!                    (2,i) = SASA P1 value for atom type i
!                    (3,i) = SASA P2 value for atom type i
!                    (4,i) = SASA P3 value for atom type i
!                    (5,i) = SASA P4 value for atom type i
!                    (6,i) = INTRINSIC BORN RADIUS
!                    (7,i) = HAWKINS SCALE FACTOR
!           INDGB  - Array of dimension (NAT) containing the atom type
!                    for each atom. This indexes into GBPARM.
!           NGBP   - Number of Generalized Born Parameters.
!
! History:  Added GB parameters for Ca2+ and Mg2+
!           Added GB paramter for Water Oxygen - Type 14
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - CLOSEMPI
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE SETUPGB

        USE SystemParam, ONLY : gbparm,indgb,itype,lubd,lcbd,nubd,ncbd,&
                              & nat,nbd,nhbd,ngbp,igb,mpijob,myproc

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        !=== VARIABLES ===!

        INTEGER :: i,j,k,l,m,itp,nh,nt,ierr
        INTEGER :: nbdt(nat),nbdn(nat),lbd(2,nbd)


        !=== Check Number of Bonds ===!

        IF ( nbd == 0 ) THEN

          IF ( myproc == 0 ) THEN
          WRITE(4,*)'ERROR: The generalized Born model does not have'
          WRITE(4,*)'any parameters for ions.'
          ENDIF

          IF ( mpijob ) CALL CLOSEMPI

          STOP

        ENDIF

        !=== NOTE - Original Tsui paper has 24 parameters ===!
        !=== Here there are 26 - Ca2+ and Mg2+ were added ===!

        SELECT CASE (igb)

          CASE (1)

            IF ( myproc == 0 ) THEN
            WRITE(4,*)'Using Tsui parameters for generalized Born.'
            ENDIF

            ngbp = 26

          CASE DEFAULT

            IF ( myproc == 0 ) THEN
            WRITE(4,*)'ERROR: igb = ',igb,' not supported.'
            ENDIF

            IF ( mpijob ) CALL CLOSEMPI

            STOP

        END SELECT

        ALLOCATE (gbparm(7,ngbp),indgb(nat))

        gbparm = 0.0d0


        !=== Make List of all bonds ===!

        j = 0

        IF ( ncbd /= 0 ) THEN

          DO i=1,ncbd

            j = j + 1

            lbd(1,j) = lcbd(1,i)
            lbd(2,j) = lcbd(2,i)

          ENDDO

        ENDIF

        IF ( nubd /= 0 ) THEN

          DO i=1,nubd

            j = j + 1

            lbd(1,j) = lubd(1,i)
            lbd(2,j) = lubd(2,i)

          ENDDO

        ENDIF

        !=== Find total number of bonds for each atom ===!

        DO i=1,nat

          nbdt(i) = 0
          nbdn(i) = 0

        ENDDO

        DO i=1,nbd

          j = lbd(1,i)
          k = lbd(2,i)

          nbdt(j) = nbdt(j) + 1
          nbdt(k) = nbdt(k) + 1

          IF ( i > nhbd ) THEN

            nbdn(j) = nbdn(j) + 1
            nbdn(k) = nbdn(k) + 1

          ENDIF

        ENDDO


        !=== Assign generalized Born types to atoms ===!

        DO i=1,nat

          ierr = 1

          itp = itype(i)
          nt = nbdt(i)
          nh = nbdn(i)

          !=== Hydrogen ===!

          IF ( itp == 1 ) THEN

            ierr = 0
            indgb(i) = 1

          ENDIF

          !=== Carbon ===!

          IF ( itp == 6 ) THEN

            ierr = 0

            IF ( nt == 4 ) THEN

              IF ( nh == 1 ) indgb(i) = 2
              IF ( nh == 2 ) indgb(i) = 3
              IF ( nh == 3 ) indgb(i) = 4
              IF ( nh == 4 ) indgb(i) = 5
              IF ( nh < 1 .or. nh > 4 ) ierr = 1

            ELSEIF ( nt == 3 ) THEN

              IF ( nh == 2 ) indgb(i) = 6
              IF ( nh == 3 ) indgb(i) = 7
              IF ( nh < 2 .or. nh > 3 ) ierr = 1

            ELSE

              ierr = 1

            ENDIF

          ENDIF

          !=== Nitrogen ===!

          IF ( itp == 7 ) THEN

            ierr = 0

            IF ( nt == 4 ) THEN

              IF ( nh == 1 ) indgb(i) = 8
              IF ( nh == 2 ) indgb(i) = 9
              IF ( nh == 3 ) indgb(i) = 10
              IF ( nh < 1 .or. nh > 3 ) ierr = 1

            ELSEIF ( nt == 3 .or. nt == 2 ) THEN

              IF ( nh == 1 ) indgb(i) = 11
              IF ( nh == 2 ) indgb(i) = 12
              IF ( nh == 3 ) indgb(i) = 13
              IF ( nh < 1 .or. nh > 3 ) ierr = 1

            ELSE

              ierr = 1

            ENDIF

          ENDIF

          !=== Oxygen ===!

          IF ( itp == 8 ) THEN

            ierr = 0

            IF ( nt == 2 ) THEN

              IF ( nh == 0 ) indgb(i) = 14  !ECD - added for H2O
              IF ( nh == 1 ) indgb(i) = 14
              IF ( nh == 2 ) indgb(i) = 15
              IF ( nh < 0 .or. nh > 2 ) ierr = 1

            ELSEIF ( nt == 1 ) THEN

              !=== Default ---> Double Bonded O ===!

              indgb(i) = 16

              !=== Check for Carboxyl Oxygen ===!

              DO j=1,nbd
              IF ( lbd(1,j) == i ) k = lbd(2,j)
              IF ( lbd(2,j) == i ) k = lbd(1,j)
              ENDDO

              !=== Is this atom carbon? If so, is it ===!
              !=== bonded to another O by a single bond ===!

              IF ( itype(k) == 6 .and. nbdt(k) > 2 ) THEN

                DO j=1,nbd

                  l = lbd(1,j)
                  m = lbd(2,j)

                  IF ( l == k .and. m /= i ) THEN
                  IF ( itype(m) == 8 .and. nbdt(m) == 1 ) indgb(i) = 17
                  ELSEIF ( m == k .and. l /= i ) THEN
                  IF ( itype(l) == 8 .and. nbdt(l) == 1 ) indgb(i) = 17
                  ENDIF

                ENDDO

              ENDIF

            ELSE

              ierr = 1

            ENDIF

          ENDIF

          !=== Phosphorous ===!

          IF ( itp == 15 ) THEN

            ierr = 0

            IF ( nh == 3 ) indgb(i) = 18
            IF ( nh == 4 ) indgb(i) = 19
            IF ( nh < 3 .or. nh > 4 ) ierr = 1

          ENDIF

          !=== Sulfer ===!

          IF ( itp == 16 ) THEN

            ierr = 0

            IF ( nh == 1 ) indgb(i) = 20
            IF ( nh == 2 ) indgb(i) = 21
            IF ( nh < 1 .or. nh > 2 ) ierr = 1

          ENDIF

          !=== Chlorine ===!

          IF ( itp == 17 ) THEN

            ierr = 0

            IF ( nh == 1 ) indgb(i) = 22
            IF ( nh /= 1 ) ierr = 1

          ENDIF

          !=== Magnesium 2+ ===!

          IF ( itp == 12 ) THEN

            ierr = 0

            IF ( nt == 0 ) indgb(i) = 25
            IF ( nt /= 0 ) ierr = 1

          ENDIF

          !=== Calcium 2+ ===!

          IF ( itp == 20 ) THEN

            ierr = 0

            IF ( nt == 0 ) indgb(i) = 26
            IF ( nt /= 0 ) ierr = 1

          ENDIF

          IF ( ierr == 1 ) THEN

            IF ( myproc == 0 ) THEN
            WRITE(4,*)'ERROR: Atom #',i
            WRITE(4,*)'       Atomic number = ',itp
            WRITE(4,*)'       Total bonds = ',nt
            WRITE(4,*)'       Non-H bonds = ',nh
            WRITE(4,*)'This atom type does not have generalized Born'
            WRITE(4,*)'parameters. Please add to SETUPGB.'
            ENDIF

            IF ( mpijob ) CALL CLOSEMPI

            STOP

          ENDIF

        ENDDO

        !=== H-C and H-O and H-S Types ===!

        DO i=1,nhbd

          j = lbd(1,i)
          k = lbd(2,i)

          itp = itype(k)

          IF ( itp == 1 ) THEN
            itp = itype(j)
            j = k
          ENDIF

          IF ( itp == 6 ) indgb(j) = 23
          IF ( itp == 8 ) indgb(j) = 24
          IF ( itp == 16) indgb(j) = 24

        ENDDO


        !=== PARAMETER TABLES ===!

        !=== SASA PARAMETERS ===!

        !=== Carbon SP3 ===!

        gbparm(1,2) =  1.70d0 + 1.40d0
        gbparm(2,2) =  7.78870d-1
        gbparm(3,2) = -2.80630d-1
        gbparm(4,2) = -1.29680d-3
        gbparm(5,2) =  3.93280d-4

        gbparm(1,3) =  1.70d0 + 1.40d0
        gbparm(2,3) =  5.64820d-1
        gbparm(3,3) = -1.96080d-1
        gbparm(4,3) = -1.02190d-3
        gbparm(5,3) =  2.65800d-4

        gbparm(1,4) =  1.70d0 + 1.40d0
        gbparm(2,4) =  2.33480d-1
        gbparm(3,4) = -7.26270d-2
        gbparm(4,4) = -2.00790d-4
        gbparm(5,4) =  7.96700d-5

        gbparm(1,5) = 1.70d0 + 1.40d0
        gbparm(2,5) = 0.0d0
        gbparm(3,5) = 0.0d0
        gbparm(4,5) = 0.0d0
        gbparm(5,5) = 0.0d0

        !=== Carbon SP2 ===!

        gbparm(1,6) =  1.70d0 + 1.40d0
        gbparm(2,6) =  5.12450d-1
        gbparm(3,6) = -1.59660d-1
        gbparm(4,6) = -1.97810d-4
        gbparm(5,6) =  1.63920d-4

        gbparm(1,7) =  1.70d0 + 1.40d0
        gbparm(2,7) =  7.03440d-2
        gbparm(3,7) = -1.90150d-2
        gbparm(4,7) = -2.20090d-5
        gbparm(5,7) =  1.68750d-5

        !=== Nitrogen SP3 ===!

        gbparm(1,8) =  1.65d0 + 1.40d0
        gbparm(2,8) =  7.86020d-2
        gbparm(3,8) = -2.91980d-1
        gbparm(4,8) = -6.53700d-4
        gbparm(5,8) =  3.62470d-4

        gbparm(1,9) =  1.65d0 + 1.40d0
        gbparm(2,9) =  2.25990d-1
        gbparm(3,9) = -3.66480d-2
        gbparm(4,9) = -1.22970d-3
        gbparm(5,9) =  8.00380d-5

        gbparm(1,10) =  1.65d0 + 1.40d0
        gbparm(2,10) =  5.14810d-2
        gbparm(3,10) = -1.26030d-2
        gbparm(4,10) = -3.20060d-4
        gbparm(5,10) =  2.47740d-5

        !=== Nitrogen SP2 ===!

        gbparm(1,11) =  1.65d0 + 1.40d0
        gbparm(2,11) =  7.35110d-1
        gbparm(3,11) = -2.21160d-1
        gbparm(4,11) = -8.91480d-4
        gbparm(5,11) =  2.52300d-4

        gbparm(1,12) =  1.65d0 + 1.40d0
        gbparm(2,12) =  4.11020d-1
        gbparm(3,12) = -1.22540d-1
        gbparm(4,12) = -7.54480d-5
        gbparm(5,12) =  1.18040d-4

        gbparm(1,13) =  1.65d0 + 1.40d0
        gbparm(2,13) =  6.25770d-2
        gbparm(3,13) = -1.78740d-2
        gbparm(4,13) = -8.31200d-5
        gbparm(5,13) =  1.98490d-5

        !=== Oxygen SP3 ===!

        gbparm(1,14) =  1.60d0 + 1.40d0
        gbparm(2,14) =  7.79140d-1
        gbparm(3,14) = -2.52620d-1
        gbparm(4,14) = -1.60560d-3
        gbparm(5,14) =  3.50710d-4

        gbparm(1,15) =  1.60d0 + 1.40d0
        gbparm(2,15) =  4.93920d-1
        gbparm(3,15) = -1.60380d-1
        gbparm(4,15) = -1.55120d-4
        gbparm(5,15) =  1.64530d-4

        !=== Oxygen SP2 ===!

        gbparm(1,16) =  1.60d0 + 1.40d0
        gbparm(2,16) =  6.85630d-1
        gbparm(3,16) = -1.86800d-1
        gbparm(4,16) = -1.35570d-3
        gbparm(5,16) =  2.37430d-4

        gbparm(1,17) =  1.60d0 + 1.40d0
        gbparm(2,17) =  8.88570d-1
        gbparm(3,17) = -3.34210d-1
        gbparm(4,17) = -1.86830d-3
        gbparm(5,17) =  4.93720d-4

        !=== Phosphorous ===!

        gbparm(1,18) =  1.90d0 + 1.40d0
        gbparm(2,18) =  3.86500d-1
        gbparm(3,18) = -1.82490d-1
        gbparm(4,18) = -3.65980d-3
        gbparm(5,18) =  4.26400d-4

        gbparm(1,19) =  1.90d0 + 1.40d0
        gbparm(2,19) =  3.87300d-2
        gbparm(3,19) = -8.93390d-3
        gbparm(4,19) =  8.35820d-6
        gbparm(5,19) =  3.03810d-6

        !=== Sulfer ===!

        gbparm(1,20) =  1.90d0 + 1.40d0
        gbparm(2,20) =  7.72200d-1
        gbparm(3,20) = -2.63930d-1
        gbparm(4,20) =  1.06290d-3
        gbparm(5,20) =  2.17900d-4

        gbparm(1,21) =  1.90d0 + 1.40d0
        gbparm(2,21) =  5.45810d-1
        gbparm(3,21) = -1.94770d-1
        gbparm(4,21) = -1.28730d-3
        gbparm(5,21) =  2.92470d-4

        !=== Chlorine ===!

        gbparm(1,22) =  1.80d0 + 1.40d0
        gbparm(2,22) =  9.83180d-1
        gbparm(3,22) = -4.04370d-1
        gbparm(4,22) =  1.12490d-4
        gbparm(5,22) =  4.99010d-4


        !=== BORN PARAMETERS ===!

        IF ( igb == 1 ) THEN

          !=== Type 1 is H-X with X /= C,O,S ===!

          gbparm(6,1) = 1.20d0 - 0.13d0
          gbparm(7,1) = 0.85d0

          !=== Types 2-7 are Carbon ===!

          DO i=2,7
          gbparm(6,i) = 1.70d0 - 0.13d0
          gbparm(7,i) = 0.72d0
          ENDDO

          !=== Types 8-13 are Nitrogen ===!

          DO i=8,13
          gbparm(6,i) = 1.55d0 - 0.13d0
          gbparm(7,i) = 0.79d0
          ENDDO

          !=== Types 14-17 are Oxygen ===!

          DO i=14,17
          gbparm(6,i) = 1.50d0 - 0.13d0
          gbparm(7,i) = 0.85d0
          ENDDO

          !=== Types 18-19 are Phosphorous ===!

          DO i=18,19
          gbparm(6,i) = 1.85d0 - 0.13d0
          gbparm(7,i) = 0.86d0
          ENDDO

          !=== Types 20-21 are Sulfer ===!

          DO i=20,21
          gbparm(6,i) = 1.80d0 - 0.13d0
          gbparm(7,i) = 0.96d0
          ENDDO

          !=== Type 22 is Chlorine ===!

          gbparm(6,22) = 1.75d0 - 0.13d0
          gbparm(7,22) = 0.96d0

          !=== Type 23 is H bonded to Carbon ===!

          gbparm(6,23) = 1.30d0 - 0.13d0
          gbparm(7,23) = 0.85d0

          !=== Type 24 is H bonded to O or S ===!

          gbparm(6,24) = 0.80d0 - 0.13d0
          gbparm(7,24) = 0.85d0

          !=== Type 25 is Magnesium 2+ ===!

          gbparm(6,25) = 1.44d0 - 0.13d0
          gbparm(7,25) = 0.80d0

          !=== Type 26 is Calcium 2+ ===!

          gbparm(6,26) = 1.72d0 - 0.13d0
          gbparm(7,26) = 0.80d0

        ENDIF

        RETURN

      END SUBROUTINE SETUPGB
