! ==============================================================================
! Subroutine: SETUPGT
! 
! Purpose: Calculates a vector basis for an irreducible representation
!          of a point group given the character table for the point
!          group.
!
! Method:  See Chapter 3 of Dykeman Ph.D. Thesis.
!
! Arguments:
!
!           GROT  - Array of dimension (3,3,NGRP) containing the rotation
!                   matricies of the point group.
!           GCHAR - Array of dimension (NCLS,NCLS) containing the character
!                   table of the point group.
!           GBAS  - Array of dimension (3*NSITE,NBAS,NROW). On OUTPUT
!                   contains a set of Basis Vectors for Irrep. IREP.
!           MGPRM - Array of dimension (NGRP). On OUTPUT contains the
!                   permutation map for the point group. Specifically,
!                   the ith rotation matrix GROT(:,:,i) takes atoms in
!                   the primitive "cell" (site 1) to site number MGPRM(i)
!           LCLS  - Array of dimension (NGRP) containing the class type
!                   for each point group matrix.
!           NSITE - Total number of unique atom copies or "sites" that
!                   can be formed by applying all NGRP rotations.
!           NGRP  - Total number of elements in the point group.
!           NBAS  - Number of Basis Vectors
!                   (i.e. Number of times Irrep. IREP appears in Hessian)
!           NCLS  - Number of unique classes or equivalently irreps of G.
!           NROW  - Number of rows or orthogonal partners
!                   (i.e. The dimension of Irrep. IREP)
!           IREP  - The Irrep. that a basis will be formed for.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - Systemraram
! Functions -
! Subroutines - RVECTOR, CLOSEMPI
!               DSYEV - LAPACK Symmetric Diagonalizer
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE SETUPGT

        USE SystemParam, ONLY : rat,grot,gchar,gbas,mgprm,lcls,nat,&
                              & nsite,ngrp,nbas,ncls,nrow,irep,iseed,&
                              & mpijob,myproc

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        !=== VARIABLES ===!

        INTEGER, DIMENSION(:,:), ALLOCATABLE :: map

        DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: a,ra
        DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE :: rv,gmat

        INTEGER :: i,j,m,n,i1,i2,i3,j1,j2,j3
        INTEGER :: icnt,igrp,ir,ierr

        DOUBLE PRECISION :: x1,x2,x3,r1,r2,r3,d,t,x,xsq
        DOUBLE PRECISION :: vrv(ngrp,ngrp),w(ngrp),wrk(3*ngrp)

        DOUBLE PRECISION, PARAMETER :: tol = 1.0d-3
        DOUBLE PRECISION, PARAMETER :: sml = 1.0d-1


        !=== Check that irep <= ncls ===!

        IF ( irep < 1 .or. irep > ncls ) THEN

          IF ( myproc == 0 ) THEN
          WRITE(4,*)'ERROR: The irrep. that you have selected is not'
          WRITE(4,*)'a valid choice. irep = ',irep
          ENDIF

          IF ( mpijob ) CALL CLOSEMPI

          STOP

        ENDIF


        !=== SECTION 1 ===!

        !=== Find nsite ===!

        n = 3 * ngrp

        ALLOCATE (a(n))

        DO i=1,ngrp

          i1 = 3 * i - 2
          i2 = i1 + 1
          i3 = i1 + 2

          x1 = grot(1,1,i) * rat(1,1) + grot(1,2,i) * rat(2,1) &
           & + grot(1,3,i) * rat(3,1)

          x2 = grot(2,1,i) * rat(1,1) + grot(2,2,i) * rat(2,1) &
           & + grot(2,3,i) * rat(3,1)

          x3 = grot(3,1,i) * rat(1,1) + grot(3,2,i) * rat(2,1) &
           & + grot(3,3,i) * rat(3,1)

          a(i1) = x1
          a(i2) = x2
          a(i3) = x3

        ENDDO

        nsite = 1

        DO i=2,ngrp

          m = 1

          i1 = 3 * i - 2
          i2 = i1 + 1
          i3 = i1 + 2

          DO j=1,nsite

            j1 = 3 * j - 2
            j2 = j1 + 1
            j3 = j1 + 2

            x1 = a(i1) - a(j1)
            x2 = a(i2) - a(j2)
            x3 = a(i3) - a(j3)

            xsq = x1 * x1 + x2 * x2 + x3 * x3
            d = DSQRT(xsq)

            IF ( d < tol ) THEN
              m = 0
              EXIT
            ENDIF

          ENDDO

          IF ( m /= 0 ) THEN

            nsite = nsite + 1

            j1 = 3 * nsite - 2
            j2 = j1 + 1
            j3 = j1 + 2

            a(j1) = a(i1)
            a(j2) = a(i2)
            a(j3) = a(i3)

          ENDIF

        ENDDO

        !=== Form Permutation Map ===!

        ALLOCATE (map(nsite,ngrp),mgprm(ngrp))

        DO igrp=1,ngrp

          DO i=1,nsite

            i1 = 3 * i - 2
            i2 = i1 + 1
            i3 = i1 + 2

            r1 = grot(1,1,igrp) * a(i1) + grot(1,2,igrp) * a(i2) &
             & + grot(1,3,igrp) * a(i3)

            r2 = grot(2,1,igrp) * a(i1) + grot(2,2,igrp) * a(i2) &
             & + grot(2,3,igrp) * a(i3)

            r3 = grot(3,1,igrp) * a(i1) + grot(3,2,igrp) * a(i2) &
             & + grot(3,3,igrp) * a(i3)

            DO j=1,nsite

              j1 = 3 * j - 2
              j2 = j1 + 1
              j3 = j1 + 2

              x1 = a(j1) - r1
              x2 = a(j2) - r2
              x3 = a(j3) - r3

              xsq = x1 * x1 + x2 * x2 + x3 * x3
              d = DSQRT(xsq)

              IF ( d < tol ) THEN

                map(i,igrp) = j
                IF ( i == 1 ) mgprm(igrp) = j

                EXIT

              ENDIF

            ENDDO

          ENDDO

        ENDDO

        DEALLOCATE (a)


        !=== SECTION 2 ===!

        !=== Find nbas and nrow ===!

        x = 0.0d0

        DO igrp=1,ngrp

          j = 0

          DO i=1,nsite
          IF ( map(i,igrp) == i ) j = j + 1
          ENDDO

          t = grot(1,1,igrp) + grot(2,2,igrp) + grot(3,3,igrp)

          t = t * DBLE(j)

          j = lcls(igrp)
          t = t * gchar(irep,j)

          x = x + t

        ENDDO

        x = x + 0.100d0
        x = x / DBLE(ngrp)

        nbas = INT(x)
        nrow = gchar(irep,1)


        !=== SECTION 3 ===!

        !=== Form Basis Vectors ===!

        n = 3 * nsite

        ALLOCATE (a(n),ra(n),rv(n,ngrp))
        ALLOCATE (gbas(n,nbas,nrow))

        IF ( nrow > 1 ) ALLOCATE (gmat(nrow,ngrp))

        !=== Character Project onto irrep. irep ===!

 1      CALL RVECTOR (a,n,iseed)

        ra = 0.0d0

        DO igrp=1,ngrp

          i = lcls(igrp)
          x = gchar(irep,i)

          DO i=1,nsite

            i1 = 3 * i - 2
            i2 = i1 + 1
            i3 = i1 + 2

            j = map(i,igrp)

            j1 = 3 * j - 2
            j2 = j1 + 1
            j3 = j1 + 2

            r1 = grot(1,1,igrp) * a(i1) + grot(1,2,igrp) * a(i2) &
             & + grot(1,3,igrp) * a(i3)

            r2 = grot(2,1,igrp) * a(i1) + grot(2,2,igrp) * a(i2) &
             & + grot(2,3,igrp) * a(i3)

            r3 = grot(3,1,igrp) * a(i1) + grot(3,2,igrp) * a(i2) &
             & + grot(3,3,igrp) * a(i3)

            ra(j1) = ra(j1) + x * r1
            ra(j2) = ra(j2) + x * r2
            ra(j3) = ra(j3) + x * r3

          ENDDO

        ENDDO

        !=== Normalize ===!

        x = 0.0d0

        DO i=1,n

          x = x + ra(i) * ra(i)

        ENDDO

        IF ( x < sml ) GOTO 1

        x = 1.0d0 / DSQRT(x)

        DO i=1,n

          a(i) = x * ra(i)
          gbas(i,1,1) = a(i)

        ENDDO

        !=== NOTE: For the case nrow > 1 and nbas > 1 ===!
        !Then vector A is a mixture of every row. Need to!
        !form a vector that transforms as a single row.!
        !Form the matrix <V|R^T(i)*R(j)|V> and diagonalize!
        !The linear combination of R(i)*V will form a vector!
        !that will transform as a single row ===!

        IF ( nrow == 1 ) GOTO 3
        IF ( nbas == 1 ) GOTO 2

        !=== Form R(i) * V ===!

        DO igrp=1,ngrp
        DO i=1,nsite

          i1 = 3 * i - 2
          i2 = i1 + 1
          i3 = i1 + 2

          j = map(i,igrp)

          j1 = 3 * j - 2
          j2 = j1 + 1
          j3 = j1 + 2

          r1 = grot(1,1,igrp) * a(i1) + grot(1,2,igrp) * a(i2) &
           & + grot(1,3,igrp) * a(i3)

          r2 = grot(2,1,igrp) * a(i1) + grot(2,2,igrp) * a(i2) &
           & + grot(2,3,igrp) * a(i3)

          r3 = grot(3,1,igrp) * a(i1) + grot(3,2,igrp) * a(i2) &
           & + grot(3,3,igrp) * a(i3)

          rv(j1,igrp) = r1
          rv(j2,igrp) = r2
          rv(j3,igrp) = r3

        ENDDO
        ENDDO

        !=== Form matrix elements ===!

        DO i=1,ngrp
        DO j=i,ngrp

          x = 0.0d0
          DO m=1,n
          x = x + rv(m,i) * rv(m,j)
          ENDDO

          vrv(i,j) = x
          vrv(j,i) = x

        ENDDO
        ENDDO

        !=== Diagonalize ===!

        i = 3 * ngrp

        CALL DSYEV ('V','U',ngrp,vrv,ngrp,w,wrk,i,ierr)

        IF ( ierr /= 0 ) THEN

          IF ( myproc == 0 ) THEN

            WRITE(4,*)'ERROR: DSYEV in SETUPGT returned and error.'
            WRITE(4,*)'DSYEV error code = ',ierr

          ENDIF

          IF ( mpijob ) CALL CLOSEMPI

          STOP

        ENDIF

        !=== Form new vector transforming as row 1 ===!

        DO i=1,n

          x = 0.0d0
          DO j=1,ngrp
          x = x + rv(i,j) * vrv(j,ngrp)
          ENDDO

          a(i) = x

        ENDDO

        !=== Normalize vector ===!

        x = 0.0d0

        DO i=1,n
        x = x + a(i) * a(i)
        ENDDO

        x = 1.0d0 / DSQRT(x)

        DO i=1,n
        a(i) = x * a(i)
        gbas(i,1,1) = a(i)
        ENDDO

 2      CONTINUE

        !=== Now find orthogonal partners ===!

        DO igrp=1,ngrp
        DO i=1,nsite

          i1 = 3 * i - 2
          i2 = i1 + 1
          i3 = i1 + 2

          j = map(i,igrp)

          j1 = 3 * j - 2
          j2 = j1 + 1
          j3 = j1 + 2

          r1 = grot(1,1,igrp) * a(i1) + grot(1,2,igrp) * a(i2) &
           & + grot(1,3,igrp) * a(i3)

          r2 = grot(2,1,igrp) * a(i1) + grot(2,2,igrp) * a(i2) &
           & + grot(2,3,igrp) * a(i3)

          r3 = grot(3,1,igrp) * a(i1) + grot(3,2,igrp) * a(i2) &
           & + grot(3,3,igrp) * a(i3)

          rv(j1,igrp) = r1
          rv(j2,igrp) = r2
          rv(j3,igrp) = r3

        ENDDO
        ENDDO

        !=== Form nrow orthonormal vectors ===!

        icnt = 1

        DO igrp=1,ngrp

          DO ir=1,icnt

            x = 0.0d0

            DO i=1,n
            x = x + rv(i,igrp) * gbas(i,1,ir)
            ENDDO

            DO i=1,n
            rv(i,igrp) = rv(i,igrp) - x * gbas(i,1,ir)
            ENDDO

          ENDDO

          x = 0.0d0

          DO i=1,n
          x = x + rv(i,igrp) * rv(i,igrp)
          ENDDO

          IF ( x > sml ) THEN

            icnt = icnt + 1

            IF ( icnt > nrow ) THEN

              IF ( myproc == 0 ) THEN
              WRITE(4,*)'ERROR: Can form more than nrow orthonormal'
              WRITE(4,*)'vectors using rotations. Check SETUPGT.'
              ENDIF

              IF ( mpijob ) CALL CLOSEMPI

              STOP

            ENDIF

            x = 1.0d0 / DSQRT(x)

            DO i=1,n
            gbas(i,1,icnt) = x * rv(i,igrp)
            ENDDO

          ENDIF

        ENDDO

        IF ( icnt /= nrow ) THEN

          IF ( myproc == 0 ) THEN
          WRITE(4,*)'ERROR: Could not construct nrow orthonormal'
          WRITE(4,*)'vectors. Number of vectors = ',icnt
          ENDIF

          IF ( mpijob ) CALL CLOSEMPI

          STOP

        ENDIF

        !== Calculate GAMMA Matrices ===!

        DO igrp=1,ngrp

          DO i=1,nsite

            i1 = 3 * i - 2
            i2 = i1 + 1
            i3 = i1 + 2

            j = map(i,igrp)

            j1 = 3 * j - 2
            j2 = j1 + 1
            j3 = j1 + 2

            r1 = grot(1,1,igrp) * a(i1) + grot(1,2,igrp) * a(i2) &
             & + grot(1,3,igrp) * a(i3)

            r2 = grot(2,1,igrp) * a(i1) + grot(2,2,igrp) * a(i2) &
             & + grot(2,3,igrp) * a(i3)

            r3 = grot(3,1,igrp) * a(i1) + grot(3,2,igrp) * a(i2) &
             & + grot(3,3,igrp) * a(i3)

            ra(j1) = r1
            ra(j2) = r2
            ra(j3) = r3

          ENDDO

          DO ir=1,nrow

            x = 0.0d0
            DO i=1,n
            x = x + ra(i) * gbas(i,1,ir)
            ENDDO

            gmat(ir,igrp) = x

          ENDDO

          !=== Normalize Gamma Matrix ===!

          x = 0.0d0

          DO ir=1,nrow
          x = x + gmat(ir,igrp) * gmat(ir,igrp)
          ENDDO

          x = 1.0d0 / DSQRT(x)

          DO ir=1,nrow
          gmat(ir,igrp) = x * gmat(ir,igrp)
          ENDDO

        ENDDO

        !=== Find the remaining basis vectors ===!

 3      CONTINUE

        icnt = 1

        DO WHILE ( icnt /= nbas )

          CALL RVECTOR (a,n,iseed)

          !=== Project onto row 1 ===!

          ra = 0.0d0

          DO igrp=1,ngrp

            i = lcls(igrp)

            IF ( nrow == 1 ) x = gchar(irep,i)
            IF ( nrow /= 1 ) x = gmat(1,igrp)

            DO i=1,nsite

            i1 = 3 * i - 2
            i2 = i1 + 1
            i3 = i1 + 2

            j = map(i,igrp)

            j1 = 3 * j - 2
            j2 = j1 + 1
            j3 = j1 + 2

            r1 = grot(1,1,igrp) * a(i1) + grot(1,2,igrp) * a(i2) &
             & + grot(1,3,igrp) * a(i3)

            r2 = grot(2,1,igrp) * a(i1) + grot(2,2,igrp) * a(i2) &
             & + grot(2,3,igrp) * a(i3)

            r3 = grot(3,1,igrp) * a(i1) + grot(3,2,igrp) * a(i2) &
             & + grot(3,3,igrp) * a(i3)

            ra(j1) = ra(j1) + x * r1
            ra(j2) = ra(j2) + x * r2
            ra(j3) = ra(j3) + x * r3

            ENDDO

          ENDDO

          !=== Gram Schmidt row 1 vectors ===!

          DO i=1,icnt

            x = 0.0d0
            DO j=1,n
            x = x + ra(j) * gbas(j,i,1)
            ENDDO

            DO j=1,n
            ra(j) = ra(j) - x * gbas(j,i,1)
            ENDDO

          ENDDO

          !=== Normalize ===!

          x = 0.0d0

          DO i=1,n
          x = x + ra(i) * ra(i)
          ENDDO

          IF ( x < sml ) CYCLE

          x = 1.0d0 / DSQRT(x)

          icnt = icnt + 1

          DO i=1,n
          a(i) = x * ra(i)
          gbas(i,icnt,1) = a(i)
          ENDDO

          !=== Find orthogonal partners ===!

          IF ( nrow == 1 ) CYCLE

          DO ir=2,nrow

            ra = 0.0d0

            DO igrp=1,ngrp

            x = gmat(ir,igrp)

            DO i=1,nsite

            i1 = 3 * i - 2
            i2 = i1 + 1
            i3 = i1 + 2

            j = map(i,igrp)

            j1 = 3 * j - 2
            j2 = j1 + 1
            j3 = j1 + 2

            r1 = grot(1,1,igrp) * a(i1) + grot(1,2,igrp) * a(i2) &
             & + grot(1,3,igrp) * a(i3)

            r2 = grot(2,1,igrp) * a(i1) + grot(2,2,igrp) * a(i2) &
             & + grot(2,3,igrp) * a(i3)

            r3 = grot(3,1,igrp) * a(i1) + grot(3,2,igrp) * a(i2) &
             & + grot(3,3,igrp) * a(i3)

            ra(j1) = ra(j1) + x * r1
            ra(j2) = ra(j2) + x * r2
            ra(j3) = ra(j3) + x * r3

            ENDDO

            ENDDO

            !=== Normalize ===!

            x = 0.0d0

            DO i=1,n
            x = x + ra(i) * ra(i)
            ENDDO

            x = 1.0d0 / DSQRT(x)

            DO i=1,n
            gbas(i,icnt,ir) = x * ra(i)
            ENDDO

          ENDDO

        ENDDO

        DEALLOCATE (map,a,ra)

        IF ( nrow > 1 ) DEALLOCATE (gmat)

        RETURN

      END SUBROUTINE SETUPGT
