! ==============================================================================
! Subroutine: SETUPTPC (T,DH,DDH,S,XI,DXI)
! 
! Purpose: Sets up the matrices and variables required for integrating
!          the equations of motion for the pressure / temperature coupling
!          algorithms of either the (1) Nose-Klein or (2) Weak coupling.
!
! Method: (1) Nose and Klein, Mol. Phys., 50 1055 (1983).
!         (2) Berendsen et al., J. Chem. Phys., 81 3684 (1984).
!
! Arguments:
!
!           T   - 3 X 3 kinetic energy tensor.
!           DH  - 3 X 3 matrix of the derivatives of the direct
!                 lattice vectors with respect to time.
!           DDH - 3 X 3 matrix of the second derivatives of the
!                 direct lattice vectors with respect to time.
!           S   - Time Scaling element in the Nose-Klein coupling
!                 algorithm.
!           XI  - Derivative of LN(s) with respect to time.
!           DXI - Second derivative of LN(s) with respect to time.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE SETUPTPC (T,DH,DDH,S,XI,DXI)

        USE SystemParam, ONLY : dlv,rlv,vir,vol,temp,btemp,pres,&
                              & comp,taut,taup,itc,ipc,ips,ipbc

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        DOUBLE PRECISION, INTENT(INOUT) :: t(3,3),dh(3,3),ddh(3,3)
        DOUBLE PRECISION, INTENT(INOUT) :: s,xi,dxi

        !=== VARIABLES ===!

        INTEGER :: i

        DOUBLE PRECISION :: x1,x2,x3,arg,wt,wp
        DOUBLE PRECISION :: wrk(3,3),dsq(3)

        DOUBLE PRECISION, PARAMETER :: cfac1 = 9.6485339840d3 
        DOUBLE PRECISION, PARAMETER :: cfac2 = 6.0221417942d-3


        wt = taut * taut * btemp

        wp = taup * taup * pres


        !=== Initialize ===!

        s = 1.0d0
        xi = 0.0d0
        dxi = 0.0d0

        dh = 0.0d0
        ddh = 0.0d0

        DO i=1,3

          arg = dlv(1,i) * dlv(1,i)
          arg = arg + dlv(2,i) * dlv(2,i)
          dsq(i) = arg + dlv(3,i) * dlv(3,i)

        ENDDO

        !=== Temperature coupling ===!

        IF ( itc == 1 ) THEN

          dxi = ( temp - btemp ) / wt

        ENDIF

        !=== Pressure coupling ===!

        IF ( ipbc == 2 ) THEN

          DO i=1,3

            wrk(1,i) = cfac1 * vir(1,i)
            wrk(2,i) = cfac1 * vir(2,i)
            wrk(3,i) = cfac1 * vir(3,i)

          ENDDO

          arg = cfac2 * pres * vol

          wrk(1,1) = wrk(1,1) - arg
          wrk(2,2) = wrk(2,2) - arg
          wrk(3,3) = wrk(3,3) - arg

        ENDIF

        !=== Nose-Klien Pressure Coupling ===!

        IF ( ipbc == 2 .and. ipc == 1 ) THEN

          DO i=1,3

            ddh(i,1) = ( 2.0d0 * t(i,1) + wrk(i,1) ) / wp
            ddh(i,2) = ( 2.0d0 * t(i,2) + wrk(i,2) ) / wp
            ddh(i,3) = ( 2.0d0 * t(i,3) + wrk(i,3) ) / wp

            IF ( ips /= 1 ) THEN

              x1 = ddh(i,1) * rlv(1,1) + ddh(i,2) * rlv(1,2) &
               & + ddh(i,3) * rlv(1,3)
              x2 = ddh(i,1) * rlv(2,1) + ddh(i,2) * rlv(2,2) &
               & + ddh(i,3) * rlv(2,3)
              x3 = ddh(i,1) * rlv(3,1) + ddh(i,2) * rlv(3,2) &
               & + ddh(i,3) * rlv(3,3)

              ddh(i,1) = x1
              ddh(i,2) = x2
              ddh(i,3) = x3

            ENDIF

          ENDDO

          IF ( ips == 1 ) THEN

            arg = ddh(1,1) + ddh(2,2) + ddh(3,3)
            arg = arg / dsq(1)

            DO i=1,3

              ddh(1,i) = dlv(1,i) * arg
              ddh(2,i) = dlv(2,i) * arg
              ddh(3,i) = dlv(3,i) * arg

            ENDDO

          ELSEIF ( ips == 2 ) THEN

            x1 = dlv(1,1) * ddh(1,1) + dlv(2,1) * ddh(2,1) &
             & + dlv(3,1) * ddh(3,1)
            x2 = dlv(1,2) * ddh(1,2) + dlv(2,2) * ddh(2,2) &
             & + dlv(3,2) * ddh(3,2)
            x3 = dlv(1,3) * ddh(1,3) + dlv(2,3) * ddh(2,3) &
             & + dlv(3,3) * ddh(3,3)

            x1 = x1 / dsq(1)
            x2 = x2 / dsq(2)
            x3 = x3 / dsq(3)

            DO i=1,3

              ddh(i,1) = dlv(i,1) * x1
              ddh(i,2) = dlv(i,2) * x2
              ddh(i,3) = dlv(i,3) * x3

            ENDDO

          ENDIF

        ENDIF

        !=== Berendsen Pressure Coupling ===!

        IF ( ipbc == 2 .and. ipc == 2 ) THEN

          arg = vol * taup * cfac2
          arg = comp / ( 3.0d0 * arg )

          DO i=1,3

            dh(i,1) = arg * ( 2.0d0 * t(i,1) + wrk(i,1) )
            dh(i,2) = arg * ( 2.0d0 * t(i,2) + wrk(i,2) )
            dh(i,3) = arg * ( 2.0d0 * t(i,3) + wrk(i,3) )

            IF ( ips /= 1 ) THEN

              x1 = dh(i,1) * dlv(1,1) + dh(i,2) * dlv(2,1) &
               & + dh(i,3) * dlv(3,1)
              x2 = dh(i,1) * dlv(1,2) + dh(i,2) * dlv(2,2) &
               & + dh(i,3) * dlv(3,2)
              x3 = dh(i,1) * dlv(1,3) + dh(i,2) * dlv(2,3) &
               & + dh(i,3) * dlv(3,3)

              dh(i,1) = x1
              dh(i,2) = x2
              dh(i,3) = x3

            ENDIF

          ENDDO

          IF ( ips == 1 ) THEN

            arg = dh(1,1) + dh(2,2) + dh(3,3)
            arg = arg / 3.0d0

            DO i=1,3

              dh(1,i) = dlv(1,i) * arg
              dh(2,i) = dlv(2,i) * arg
              dh(3,i) = dlv(3,i) * arg

            ENDDO

          ELSEIF ( ips == 2 ) THEN

            x1 = dlv(1,1) * dh(1,1) + dlv(2,1) * dh(2,1) &
             & + dlv(3,1) * dh(3,1)
            x2 = dlv(1,2) * dh(1,2) + dlv(2,2) * dh(2,2) &
             & + dlv(3,2) * dh(3,2)
            x3 = dlv(1,3) * dh(1,3) + dlv(2,3) * dh(2,3) &
             & + dlv(3,3) * dh(3,3)

            x1 = x1 / dsq(1)
            x2 = x2 / dsq(2)
            x3 = x3 / dsq(3)

            DO i=1,3

              dh(i,1) = dlv(i,1) * x1
              dh(i,2) = dlv(i,2) * x2
              dh(i,3) = dlv(i,3) * x3

            ENDDO

          ENDIF

        ENDIF

        RETURN

      END SUBROUTINE SETUPTPC
