! ==============================================================================
! Subroutine: MDYNAMICS
! 
! Purpose: Performs a classical molecular dynamics simulation in one
!          of the following ensembles: (1) VACUUM (2) NVT (3) NPT
!          (4) NVE (5) NPH.
!
! Method: Integrates the equations of motion using a velocity
!         Verlet scheme:
!
!         R(t+dt) = R(t) + dt * [ V(t) + dt/2 * A(t) ]
!         V(t+dt) = V(t) + dt/2 * [ A(t) + A(t+dt) ]
!
! Arguments:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - ENERGY, SHAKE, RATTLE, SCALER, SCALEV, SETUPTPC
!               EWALDSETUP, PMESETUP, OUTPUTMD, NEIGHBORMAP
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE MDYNAMICS

        USE SystemParam, ONLY : rat,vel,amass,dlv,tstep,temp,time,&
                              & nat,ncbd,nsbk,nsmap,nsteps,ipbc,itc,&
                              & ipc,ishk,itype,myproc,rstfile

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        !=== VARIABLES ===!

        INTEGER :: i,j,istep

        CHARACTER (LEN=60) :: fmat

        DOUBLE PRECISION :: eb,ea,ed,ec,ev,ek,x1,x2,x3
        DOUBLE PRECISION :: s,xi,dxi,c,ct,ck,dh(3,3),ddh(3,3)
        DOUBLE PRECISION :: fold(3,nat),fnew(3,nat),t(3,3)

        DOUBLE PRECISION, PARAMETER :: cfac = 9.6485339840d3
        DOUBLE PRECISION, PARAMETER :: boltz = 8.61734315d-5


        !=== Constants ===!

        ct = tstep * cfac
        ct = 0.50d0 * ct

        i = 3 * nat - ncbd

        ck = DBLE(i) * boltz
        ck = 2.0d0 / ck

        !=== Initalize ===!

        ek = 0.0d0
        t = 0.0d0

        CALL ENERGY (eb,ea,ed,ec,ev,fnew)

        DO i=1,nat

          c = 0.50d0 * amass(i)

          fold(1,i) = fnew(1,i)
          fold(2,i) = fnew(2,i)
          fold(3,i) = fnew(3,i)

          x1 = c * vel(1,i) * vel(1,i)
          x2 = c * vel(2,i) * vel(2,i)
          x3 = c * vel(3,i) * vel(3,i)

          ek = ek + x1 + x2 + x3

          IF ( ipbc /= 2 ) CYCLE

          t(1,1) = t(1,1) + x1
          t(2,2) = t(2,2) + x2
          t(3,3) = t(3,3) + x3

          x1 = c * vel(1,i) * vel(2,i)
          x2 = c * vel(1,i) * vel(3,i)
          x3 = c * vel(2,i) * vel(3,i)

          t(2,1) = t(2,1) + x1
          t(3,1) = t(3,1) + x2
          t(3,2) = t(3,2) + x3
          t(1,2) = t(1,2) + x1
          t(1,3) = t(1,3) + x2
          t(2,3) = t(2,3) + x3

        ENDDO

        ek = ek / cfac
        temp = ek * ck

        IF ( itc /= 0 .or. ipbc == 2 ) THEN

          CALL SETUPTPC (t,dh,ddh,s,xi,dxi)

        ENDIF


        !=== MD Routine ===!

        DO istep=1,nsteps

          !=== Scale Atomic Positions ===!

          IF ( ipbc == 2 ) THEN

            CALL SCALER (dh,ddh,s,xi,dxi)

          ENDIF

          !=== SHAKE Bond Constraints ===!

          IF ( ishk /= 0 .and. ncbd /= 0 ) THEN

            CALL SHAKE (fold,fnew)

          ENDIF

          !=== Verlet Position Step ===!

          DO i=1,nat

            c = ct / amass(i)

            x1 = vel(1,i) + c * fnew(1,i)
            x2 = vel(2,i) + c * fnew(2,i)
            x3 = vel(3,i) + c * fnew(3,i)

            rat(1,i) = rat(1,i) + tstep * x1
            rat(2,i) = rat(2,i) + tstep * x2
            rat(3,i) = rat(3,i) + tstep * x3

          ENDDO

          !=== New Force ===!

          CALL ENERGY (eb,ea,ed,ec,ev,fnew)

          !=== RATTLE Velocity Constraints ===!

          IF ( ishk /= 0 .and. ncbd /= 0 ) THEN

            CALL RATTLE (fold,fnew)

          ENDIF

          !=== Verlet Velocity Step ===!

          ek = 0.0d0
          t = 0.0d0

          DO i=1,nat

            c = ct / amass(i)

            x1 = c * ( fold(1,i) + fnew(1,i) )
            x2 = c * ( fold(2,i) + fnew(2,i) )
            x3 = c * ( fold(3,i) + fnew(3,i) )

            vel(1,i) = vel(1,i) + x1
            vel(2,i) = vel(2,i) + x2
            vel(3,i) = vel(3,i) + x3

            fold(1,i) = fnew(1,i)
            fold(2,i) = fnew(2,i)
            fold(3,i) = fnew(3,i)

            c = 0.50d0 * amass(i)

            x1 = c * vel(1,i) * vel(1,i)
            x2 = c * vel(2,i) * vel(2,i)
            x3 = c * vel(3,i) * vel(3,i)

            ek = ek + x1 + x2 + x3

            IF ( ipbc /= 2 ) CYCLE

            t(1,1) = t(1,1) + x1
            t(2,2) = t(2,2) + x2
            t(3,3) = t(3,3) + x3

            x1 = c * vel(1,i) * vel(2,i)
            x2 = c * vel(1,i) * vel(3,i)
            x3 = c * vel(2,i) * vel(3,i)

            t(2,1) = t(2,1) + x1
            t(3,1) = t(3,1) + x2
            t(3,2) = t(3,2) + x3
            t(1,2) = t(1,2) + x1
            t(1,3) = t(1,3) + x2
            t(2,3) = t(2,3) + x3

          ENDDO

          ek = ek / cfac
          temp = ek * ck

          !=== Scale Velocities ===!

          IF ( itc /= 0 .or. ipbc == 2 ) THEN

            CALL SCALEV (t,dh,ddh,s,xi,dxi,ek,fold,fnew)

          ENDIF

          !=== Output Data ===!

          time = time + tstep

          IF ( myproc == 0 ) THEN

            CALL OUTPUTMD (ek,eb,ea,ed,ec,ev,istep)

          ENDIF

          !=== Update Neighbors ===!

          j = MODULO(istep,nsmap)

          IF ( j == 0 ) CALL NEIGHBORMAP

          !=== Write Backup File ===!

          j = 1

          IF ( nsbk /= 0 ) j = MODULO(istep,nsbk)

          IF ( j == 0 .and. myproc == 0 ) THEN

            fmat = '(I8,E16.8)'
            WRITE(10,fmat)nat,time

            fmat = '(I3,6F13.7)'
            WRITE(10,fmat)(itype(i),rat(1,i),rat(2,i),rat(3,i),&
                         & vel(1,i),vel(2,i),vel(3,i),i=1,nat)

            IF ( ipbc /= 0 ) THEN

              fmat = '(3F13.7)'
              WRITE(10,fmat)dlv(1,1),dlv(1,2),dlv(1,3)
              WRITE(10,fmat)dlv(2,1),dlv(2,2),dlv(2,3)
              WRITE(10,fmat)dlv(3,1),dlv(3,2),dlv(3,3)

            ENDIF

            CLOSE (UNIT = 10)

            OPEN (UNIT = 10,FILE = rstfile,STATUS = 'OLD')

          ENDIF

        ENDDO

        !=== Write Final RE-start File ===!

        IF ( myproc == 0 ) THEN

          fmat = '(I8,E16.8)'
          WRITE(10,fmat)nat,time

          fmat = '(I3,6F13.7)'
          WRITE(10,fmat)(itype(i),rat(1,i),rat(2,i),rat(3,i),&
                       & vel(1,i),vel(2,i),vel(3,i),i=1,nat)

          IF ( ipbc /= 0 ) THEN

            fmat = '(3F13.7)'
            WRITE(10,fmat)dlv(1,1),dlv(1,2),dlv(1,3)
            WRITE(10,fmat)dlv(2,1),dlv(2,2),dlv(2,3)
            WRITE(10,fmat)dlv(3,1),dlv(3,2),dlv(3,3)

          ENDIF

        ENDIF

        RETURN

      END SUBROUTINE MDYNAMICS
