! ==============================================================================
! Subroutine: MINIMIZE
! 
! Purpose: Performs a minimization of the energy functional to find
!          a local minimum.
!
! Method: One of the following methods is used:
!
!         (1) Power Quench - Velocites are set to zero if F.V < 0
!         (2) Global Power Quench
!         (3) Steepest Decents
!         (4) Conjugate Gradents
!
! Arguments:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - ENERGY, LINEMIN, NEIGHBORMAP, SYMMETRYMAP,
!               OUTPUTMIN, CLOSEMPI
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE MINIMIZE

        USE SystemParam, ONLY : rat,vel,amass,dlv,tolf,tstep,nat,&
                              & ipbc,imin,isym,nsbk,nsteps,nsmap,&
                              & itype,myproc,mpijob,rstfile

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        !=== VARIABLES ===!

        INTEGER :: i,j,istep,jstep

        CHARACTER (LEN=60) :: fmat

        DOUBLE PRECISION :: eb,ea,ed,ec,ev,ek,enew,eold,gam
        DOUBLE PRECISION :: x1,x2,x3,c,ct,df,dff,rmax,vmax,frms
        DOUBLE PRECISION :: fold(3,nat),fnew(3,nat)
        DOUBLE PRECISION :: pold(3,nat),pnew(3,nat)

        DOUBLE PRECISION, PARAMETER :: cfac = 9.6485339840d3 


        !=== Constants ===!

        ct = tstep * cfac
        ct = 0.50d0 * ct

        rmax = 0.40d0
        vmax = 0.25d0 * rmax / tstep

        !=== VARIOUS MINIMIZATION ROUTINES ===!

        !iMIN = 1 --> Power Quench Minimization!
        !iMIN = 2 --> Global Power Quench!
        !iMIN = 3 --> Steepest Decents Minimization!
        !iMIN = 4 --> Conjugate Gradients!


        jstep = 0

        !=== Minimization Routine ===!

        IF ( imin == 1 .or. imin == 2 ) THEN

          CALL ENERGY (eb,ea,ed,ec,ev,fold)

          eold = eb + ea + ed + ec + ev

          !=== Check Forces and Zero Velocities ===!

          df = 0.0d0

          DO i=1,nat

            df = df + fold(1,i) * fold(1,i)
            df = df + fold(2,i) * fold(2,i)
            df = df + fold(3,i) * fold(3,i)

            vel(1,i) = 0.0d0
            vel(2,i) = 0.0d0
            vel(3,i) = 0.0d0

          ENDDO

          frms = df / DBLE(nat)
          frms = DSQRT(frms)


          DO istep=1,nsteps

            jstep = jstep + 1

            !=== Verlet Position Step ===!

            DO i=1,nat

              c = ct / amass(i)

              x1 = vel(1,i) + c * fold(1,i)
              x2 = vel(2,i) + c * fold(2,i)
              x3 = vel(3,i) + c * fold(3,i)

              x1 = tstep * x1
              x2 = tstep * x2
              x3 = tstep * x3

              IF ( DABS(x1) > rmax ) x1 = DSIGN(rmax,x1)
              IF ( DABS(x2) > rmax ) x2 = DSIGN(rmax,x2)
              IF ( DABS(x3) > rmax ) x3 = DSIGN(rmax,x3)

              rat(1,i) = rat(1,i) + x1
              rat(2,i) = rat(2,i) + x2
              rat(3,i) = rat(3,i) + x3

            ENDDO

            !=== New Force ===!

            CALL ENERGY (eb,ea,ed,ec,ev,fnew)

            enew = eb + ea + ed + ec + ev

            !=== Verlet Velocity Step ===!

            ek = 0.0d0
            df = 0.0d0

            DO i=1,nat

              c = ct / amass(i)

              x1 = c * ( fold(1,i) + fnew(1,i) )
              x2 = c * ( fold(2,i) + fnew(2,i) )
              x3 = c * ( fold(3,i) + fnew(3,i) )

              x1 = vel(1,i) + x1
              x2 = vel(2,i) + x2
              x3 = vel(3,i) + x3

              IF ( DABS(x1) > vmax ) x1 = DSIGN(vmax,x1)
              IF ( DABS(x2) > vmax ) x2 = DSIGN(vmax,x2)
              IF ( DABS(x3) > vmax ) x3 = DSIGN(vmax,x3)

              vel(1,i) = x1
              vel(2,i) = x2
              vel(3,i) = x3

              c = fnew(1,i) * vel(1,i) + &
                & fnew(2,i) * vel(2,i) + &
                & fnew(3,i) * vel(3,i)

              IF ( imin == 2 .and. eold < enew ) c = -1.0d0

              IF ( c < 0.0d0 ) THEN
                vel(1,i) = 0.0d0
                vel(2,i) = 0.0d0
                vel(3,i) = 0.0d0
              ENDIF

              fold(1,i) = fnew(1,i)
              fold(2,i) = fnew(2,i)
              fold(3,i) = fnew(3,i)

              c = 0.50d0 * amass(i)

              x1 = c * vel(1,i) * vel(1,i)
              x2 = c * vel(2,i) * vel(2,i)
              x3 = c * vel(3,i) * vel(3,i)

              ek = ek + x1 + x2 + x3

              df = df + fold(1,i) * fold(1,i)
              df = df + fold(2,i) * fold(2,i)
              df = df + fold(3,i) * fold(3,i)

            ENDDO

            ek = ek / cfac
            eold = enew

            frms = df / DBLE(nat)
            frms = DSQRT(frms)

            IF ( frms <= tolf ) EXIT

            !=== Output Data ===!

            IF ( myproc == 0 ) THEN

              CALL OUTPUTMIN (ek,eb,ea,ed,ec,ev,frms,nat,istep)

            ENDIF

            !=== Update Neighbors ===!

            j = MODULO(istep,nsmap)

            IF ( j == 0 ) THEN

              IF ( isym == 0 ) CALL NEIGHBORMAP
              IF ( isym == 1 ) CALL SYMMETRYMAP

            ENDIF

            !=== Write Backup File ===!

            j = 1

            IF ( nsbk /= 0 ) j = MODULO(istep,nsbk)

            IF ( j == 0 .and. myproc == 0 ) THEN

              fmat = '(I8,E16.8)'
              WRITE(10,fmat)nat,0.0d0

              fmat = '(I3,3F13.7)'
              WRITE(10,fmat)(itype(i),rat(1,i),rat(2,i),&
                           & rat(3,i),i=1,nat)

              IF ( ipbc /= 0 ) THEN

                fmat = '(3F13.7)'
                WRITE(10,fmat)dlv(1,1),dlv(1,2),dlv(1,3)
                WRITE(10,fmat)dlv(2,1),dlv(2,2),dlv(2,3)
                WRITE(10,fmat)dlv(3,1),dlv(3,2),dlv(3,3)

              ENDIF

              CLOSE (UNIT = 10)

              OPEN (UNIT = 10,FILE = rstfile,STATUS = 'OLD')

            ENDIF

          ENDDO

        ELSEIF ( imin == 3 .or. imin == 4 ) THEN

          ek = 0.0d0

          !=== Calculate First Search Direction ===!

          CALL ENERGY (eb,ea,ed,ec,ev,fold)

          df = 0.0d0

          DO i=1,nat

            pold(1,i) = fold(1,i)
            pold(2,i) = fold(2,i)
            pold(3,i) = fold(3,i)

            pnew(1,i) = pold(1,i)
            pnew(2,i) = pold(2,i)
            pnew(3,i) = pold(3,i)

            fnew(1,i) = fold(1,i)
            fnew(2,i) = fold(2,i)
            fnew(3,i) = fold(3,i)

            df = df + fold(1,i) * fold(1,i)
            df = df + fold(2,i) * fold(2,i)
            df = df + fold(3,i) * fold(3,i)

          ENDDO

          frms = df / DBLE(nat)
          frms = DSQRT(frms)

          gam = 1.0d0 / df

          DO istep=1,nsteps

            jstep = jstep + 1

            !=== Line Minimization ===!

            IF ( imin == 3 ) THEN

              CALL LINEMIN (fnew)

            ELSEIF ( imin == 4 ) THEN

              CALL LINEMIN (pnew)

            ENDIF

            !=== New Forces ===!

            CALL ENERGY (eb,ea,ed,ec,ev,fnew)

            !=== Calculate Gradient Dot Products ===!

            df = 0.0d0
            dff = 0.0d0

            DO i=1,nat

              df = df + fnew(1,i) * fnew(1,i)
              df = df + fnew(2,i) * fnew(2,i)
              df = df + fnew(3,i) * fnew(3,i)

              x1 = fnew(1,i) - fold(1,i)
              x2 = fnew(2,i) - fold(2,i)
              x3 = fnew(3,i) - fold(3,i)

              dff = dff + x1 * fnew(1,i)
              dff = dff + x2 * fnew(2,i)
              dff = dff + x3 * fnew(3,i)

            ENDDO

            frms = df / DBLE(nat)
            frms = DSQRT(frms)

            IF ( frms <= tolf ) EXIT

            !=== Get New Search Direction ===!

            gam = gam * dff

            DO i=1,nat

              pnew(1,i) = fnew(1,i) + gam * pold(1,i)
              pnew(2,i) = fnew(2,i) + gam * pold(2,i)
              pnew(3,i) = fnew(3,i) + gam * pold(3,i)

              pold(1,i) = pnew(1,i)
              pold(2,i) = pnew(2,i)
              pold(3,i) = pnew(3,i)

              fold(1,i) = fnew(1,i)
              fold(2,i) = fnew(2,i)
              fold(3,i) = fnew(3,i)

            ENDDO

            gam = 1.0d0 / df

            !=== Output Data ===!

            IF ( myproc == 0 ) THEN

              CALL OUTPUTMIN (ek,eb,ea,ed,ec,ev,frms,nat,istep)

            ENDIF

            !=== Update Neighbors ===!

            j = MODULO(istep,nsmap)

            IF ( j == 0 ) THEN

              IF ( isym == 0 ) CALL NEIGHBORMAP
              IF ( isym == 1 ) CALL SYMMETRYMAP

            ENDIF

            !=== Write Backup File ===!

            j = 1

            IF ( nsbk /= 0 ) j = MODULO(istep,nsbk)

            IF ( j == 0 .and. myproc == 0 ) THEN

              fmat = '(I8,E16.8)'
              WRITE(10,fmat)nat,0.0d0

              fmat = '(I3,3F13.7)'
              WRITE(10,fmat)(itype(i),rat(1,i),rat(2,i),&
                           & rat(3,i),i=1,nat)

              IF ( ipbc /= 0 ) THEN

                fmat = '(3F13.7)'
                WRITE(10,fmat)dlv(1,1),dlv(1,2),dlv(1,3)
                WRITE(10,fmat)dlv(2,1),dlv(2,2),dlv(2,3)
                WRITE(10,fmat)dlv(3,1),dlv(3,2),dlv(3,3)

              ENDIF

              CLOSE (UNIT = 10)

              OPEN (UNIT = 10,FILE = rstfile,STATUS = 'OLD')

            ENDIF

          ENDDO

        ELSE

          IF ( myproc == 0 ) THEN

            WRITE(4,*)'ERROR: imin must equal 1 to 4'
            WRITE(4,*)'imin = ',iMIN

          ENDIF

          IF ( mpijob ) CALL CLOSEMPI

          STOP

        ENDIF

        IF ( myproc == 0 .and. frms > tolf ) THEN

          WRITE(4,*)'WARNING: Mimimum not found -- no success.'

        ENDIF

        IF ( myproc == 0 .and. frms <= tolf ) THEN

          WRITE(4,*)'Minimum found -- success.'

        ENDIF

        IF ( myproc == 0 ) THEN

          WRITE(4,*)'Number of minimization steps = ',jstep
          WRITE(4,*)'Magnitude of RMS gradient F = ',frms

        ENDIF

        !=== Write Final RE-start File ===!

        IF ( myproc == 0 ) THEN

          fmat = '(I8,E16.8)'
          WRITE(10,fmat)nat,0.0d0

          fmat = '(I3,3F13.7)'
          WRITE(10,fmat)(itype(i),rat(1,i),rat(2,i),rat(3,i),i=1,nat)

          IF ( ipbc /= 0 ) THEN

            fmat = '(3F13.7)'
            WRITE(10,fmat)dlv(1,1),dlv(1,2),dlv(1,3)
            WRITE(10,fmat)dlv(2,1),dlv(2,2),dlv(2,3)
            WRITE(10,fmat)dlv(3,1),dlv(3,2),dlv(3,3)

          ENDIF

        ENDIF

        RETURN

      END SUBROUTINE MINIMIZE
