! ==============================================================================
! Subroutine: VIBES
! 
! Purpose: Calculates the eigenvectors and eigenvalues of a large
!          system of atoms and molecules which interact via a
!          classical potential energy function.
!
! Method: Uses the Phonon Functional Method of Dykeman and Sankey
!
!         E.C. Dykeman and O.F. Sankey,
!         Phys. Rev. Lett. 100, 028101 (2008)
!         E.C. Dykeman and O.F. Sankey,
!         J. Phys: Condens. Mat. 21, 035116 (2009)
!
! Arguments:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - HESSIANNBP, DYNAMICALU, GDYNAMICALU,
!               OUTPUTVIBE, CLOSEMPI
!               DSTERF - LAPACK Diagonalizer
!               DSYGV - LAPACK General Symmetric EV solver
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE VIBES

        USE SystemParam, ONLY : rat,amass,gbas,tolf,nat,nsteps,nev,&
                              & nsbk,nbas,nrow,nsite,isym,iseed,&
                              & myproc,mpijob,rstfile

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        !=== VARIABLES ===!

        DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: wrk1,wrk2
        DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: bk,bkp,bkm
        DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE :: u,p,du,dp,g

        INTEGER :: i,j,k,m,istep,ierr,nmat,nrec

        CHARACTER (LEN=60) :: fmat

        DOUBLE PRECISION :: e,q,r,a,b,c,d,a1,a2,a3,df,dfn
        DOUBLE PRECISION :: x,y,wmax,xlam,gam,xn,sm(nat)
        DOUBLE PRECISION :: spp(nev,nev),suu(nev,nev),spu(nev,nev)
        DOUBLE PRECISION :: pdp(nev,nev),udu(nev,nev),pdu(nev,nev)

        INTEGER, PARAMETER :: nrst = 100

        DOUBLE PRECISION, PARAMETER :: cfac = 0.52147091913746591862d3
        DOUBLE PRECISION, PARAMETER :: pi = 3.14159265358979323846d0
        DOUBLE PRECISION, PARAMETER :: tolw = 1.000000000000000000d0


        !=== Setup and Checks ===!

        DO i=1,nat

          x = amass(i)
          sm(i) = 1.0d0 / DSQRT(x)

        ENDDO

        nmat = 3 * nat

        IF ( isym == 1 ) nmat = nbas * nat

        IF ( nev > nmat ) THEN

          IF ( myproc == 0 ) THEN
          WRITE(4,*)'ERROR: The number of eigenvectors that have'
          WRITE(4,*)'been requested is greater than the dimension'
          WRITE(4,*)'of the dynamical matrix, N = ',nmat
          ENDIF

          IF ( mpijob ) CALL CLOSEMPI

          STOP

        ENDIF

        IF ( nmat == 1 ) THEN

          u(1,1) = 1.0d0

          GOTO 2

        ENDIF

        !=== Calculate non-bonded Hessian terms ===!

        CALL HESSIANNBP

        !=== Allocate Storage ===!

        ALLOCATE (bk(nmat),bkp(nmat),bkm(nmat))
        ALLOCATE (u(nmat,nev),p(nmat,nev),du(nmat,nev))
        ALLOCATE (dp(nmat,nev),g(nmat,nev))


        !=== Read in Old Vectors ===!

        DO i=1,nev

          CALL RVECTOR (u(:,i),nmat,iseed)

        ENDDO

        IF ( myproc /= 0 ) THEN

          OPEN (UNIT = 10, FILE = rstfile, STATUS = 'Unknown')

        ENDIF

        READ(10,*,END = 1)i

        fmat = '(3E24.16)'

        DO j=1,nev

          READ(10,*)x
          READ(10,fmat)(u(i,j),i=1,nmat)

        ENDDO

        IF ( isym == 1 ) THEN

          fmat = '(3I6)'
          READ(10,fmat)nsite,nbas,nrow

          IF ( ALLOCATED(gbas) ) DEALLOCATE (gbas)

          ALLOCATE (gbas(3*nsite,nbas,nrow))

          fmat = '(3E24.16)'

          DO k=1,nrow
          DO j=1,nbas

            READ(10,fmat)(gbas(i,j,k),i=1,3*nsite)

          ENDDO
          ENDDO

        ENDIF

 1      CONTINUE

        IF ( myproc == 0 ) REWIND (UNIT = 10)
        IF ( myproc /= 0 ) CLOSE (UNIT = 10)


        !=== Calculate WMAX ===!

        nrec = 10
        IF ( nmat <= nrec ) nrec = nmat

        ALLOCATE (wrk1(nrec),wrk2(nrec-1))

        CALL RVECTOR (bk,nmat,iseed)

        DO i=1,nrec

          IF ( isym == 0 ) THEN
          CALL DYNAMICALU (bk,bkp,sm)
          ELSE
          CALL GDYNAMICALU (bk,bkp,sm)
          ENDIF

          x = 0.0d0

          DO j=1,nmat
          x = x + bk(j) * bkp(j)
          ENDDO

          wrk1(i) = x

          DO j=1,nmat

            bkp(j) = bkp(j) - x * bk(j)

            IF ( i > 1 ) THEN
            bkp(j) = bkp(j) - y * bkm(j)
            ENDIF

          ENDDO

          y = 0.0d0

          DO j=1,nmat
          y = y + bkp(j) * bkp(j)
          ENDDO

          y = DSQRT(y)

          IF ( i < nrec ) wrk2(i) = y

          x = 1.0d0 / y

          DO j=1,nmat
          bkm(j) = bk(j)
          bk(j) = x * bkp(j)
          ENDDO

        ENDDO

        CALL DSTERF (nrec,wrk1,wrk2,ierr)

        IF ( ierr /= 0 ) THEN

          IF ( myproc == 0 ) THEN
          WRITE(4,*)'ERROR: DSTERF in VIBES failed.'
          WRITE(4,*)'DSTERF returned error code ierr = ',ierr
          ENDIF

          IF ( mpijob ) CALL CLOSEMPI

          STOP

        ENDIF

        wmax = wrk1(nrec)
        wmax = wmax + tolw

        DEALLOCATE (wrk1,wrk2)


        !=== Calculate (D-WMAX)|Ui> ===!

        DO i=1,nev

          IF ( isym == 0 ) THEN
          CALL DYNAMICALU (u(:,i),du(:,i),sm)
          ELSE
          CALL GDYNAMICALU (u(:,i),du(:,i),sm)
          ENDIF

          DO j=1,nmat
          du(j,i) = du(j,i) - wmax * u(j,i)
          ENDDO

        ENDDO

        !=== Calculate <u|u> and <u|D|u> overlaps ===!

        DO i=1,nev
        DO j=i,nev

          suu(i,j) = 0.0d0
          udu(i,j) = 0.0d0

          DO k=1,nmat
          suu(i,j) = suu(i,j) + u(k,i) * u(k,j)
          udu(i,j) = udu(i,j) + u(k,i) * du(k,j)
          ENDDO

          suu(j,i) = suu(i,j)
          udu(j,i) = udu(i,j)

        ENDDO
        ENDDO

        !=== Calculate Phonon Functional Energy ===!

        e = 0.0d0

        DO i=1,nev

          x = 2.0d0 - suu(i,i)
          e = e + x * udu(i,i)

          DO j=i+1,nev
          e = e - 2.0d0 * suu(i,j) * udu(i,j)
          ENDDO

          DO j=1,nmat
          g(j,i) = 0.0d0
          ENDDO

        ENDDO


        !=== Start Phonon Functional Method ===!

        DO istep=1,nsteps

          !=== Restart Conjugate Gradient ===!

          i = MODULO (istep,nrst)

          IF ( i == 0 .or. istep == 1 ) THEN

            df = 1.0d0
            xn = 0.0d0

          ENDIF

          !=== Calculate Gradient ===!

          dfn = 0.0d0
          gam = 0.0d0

          DO i=1,nev
          DO k=1,nmat

            x = 2.0d0 * du(k,i)

            DO j=1,nev
            x = x - du(k,j) * suu(j,i)
            x = x - u(k,j) * udu(j,i)
            ENDDO

            x = 2.0d0 * x

            dfn = dfn + x * x

            !=== Poliak Ribere ===!

            gam = gam + x * ( x - g(k,i) )
            g(k,i) = x

            !=== Flecher Reaves ===!

!            gam = gam + x * x
!            g(k,i) = x

          ENDDO
          ENDDO

          IF ( DSQRT(dfn) < tolf ) EXIT

          !=== Calculate new search direction ===!

          gam = gam / df
          gam = gam * xn

          df = dfn

          xn = df + gam * gam
          xn = DSQRT(xn)
          x = 1.0d0 / xn

          DO i=1,nev
          DO j=1,nmat

            p(j,i) = -g(j,i) + gam * p(j,i)

            !=== NORMALIZE SEARCH DIRECTION ===!

            p(j,i) = x * p(j,i)

          ENDDO
          ENDDO

          !=== Calculate (D-WMAX)|Pi> ===!

          DO i=1,nev

            IF ( isym == 0 ) THEN
            CALL DYNAMICALU (p(:,i),dp(:,i),sm)
            ELSE
            CALL GDYNAMICALU (p(:,i),dp(:,i),sm)
            ENDIF

            DO j=1,nmat
            dp(j,i) = dp(j,i) - wmax * p(j,i)
            ENDDO

          ENDDO

          !=== Caluclate <p|p> and <p|D|p> overlaps ===!

          DO i=1,nev
          DO j=i,nev

            spp(i,j) = 0.0d0  !<Pi|Pj>!
            pdp(i,j) = 0.0d0  !<Pi|D|Pj>!

            spu(i,j) = 0.0d0  !1/2(<Pi|Uj> + <Pj|Ui>)!
            pdu(i,j) = 0.0d0  !1/2(<Pi|D|Uj> + <Pj|D|Ui>)!

            DO k=1,nmat
            spp(i,j) = spp(i,j) + p(k,i) * p(k,j)
            pdp(i,j) = pdp(i,j) + p(k,i) * dp(k,j)
            spu(i,j) = spu(i,j) + p(k,i) * u(k,j)
            spu(i,j) = spu(i,j) + p(k,j) * u(k,i)
            pdu(i,j) = pdu(i,j) + p(k,i) * du(k,j)
            pdu(i,j) = pdu(i,j) + p(k,j) * du(k,i)
            ENDDO

            spu(i,j) = 0.50d0 * spu(i,j)
            pdu(i,j) = 0.50d0 * pdu(i,j)

            spp(j,i) = spp(i,j)
            spu(j,i) = spu(i,j)
            pdp(j,i) = pdp(i,j)
            pdu(j,i) = pdu(i,j)

          ENDDO
          ENDDO

          !=== SOLVE CUBIC EQUATION ===!
          !A*X**3 + B*X**2 + C*X + D = 0!

          a = 0.0d0
          b = 0.0d0
          c = 0.0d0
          d = 0.0d0

          DO i=1,nev
          c = c + pdp(i,i)
          d = d + pdu(i,i)
          ENDDO

          c = -2.0d0 * c
          d = -2.0d0 * d

          DO i=1,nev
          DO j=1,nev

            a = a + spp(i,j) * pdp(i,j)

            b = b + spu(i,j) * pdp(i,j)
            b = b + spp(i,j) * pdu(i,j)

            c = c + 4.0d0 * spu(i,j) * pdu(i,j)
            c = c + suu(i,j) * pdp(i,j)
            c = c + spp(i,j) * udu(i,j)

            d = d + spu(i,j) * udu(i,j)
            d = d + suu(i,j) * pdu(i,j)

          ENDDO
          ENDDO

          a = 4.0d0 * a
          b = 6.0d0 * b
          c = 2.0d0 * c
          d = 2.0d0 * d

          a1 = b / a
          a2 = c / a
          a3 = d / a

          q = ( a1 * a1 - 3.0d0 * a2 )
          q = q / 9.0d0

          r = a1 * ( q - a2 / 6.0d0 )
          r = r / 3.0d0 + 0.50d0 * a3

          x = q * q * q - r * r

          IF ( x < 0.0d0 ) THEN

            !=== one root ===!

            y = DSIGN(1.0d0,r)
            x = DSQRT(-x)

            x = x + DABS(r)
            x = x**(1.0d0/3.0d0)

            xlam = -y * ( x + q / x ) - a1 / 3.0d0

          ELSE

            !=== three roots ===!

            IF ( q == 0.0d0 ) THEN

              IF ( myproc == 0 ) THEN
              WRITE(4,*)'ERROR: Failed to find root in VIBES.'
              WRITE(4,*)'Try restarting using current vectors.'
              ENDIF

              EXIT

            ENDIF

            x = q * q * q
            x = r / DSQRT(x)
            x = DACOS(x)

            q = -2.0d0 * DSQRT(q)
            r = a1 / 3.0d0

            y = q * DCOS(x / 3.0d0) - r
            a1 = a * y / 4.0d0 + b / 3.0d0
            a1 = a1 * y + c / 2.0d0
            a1 = a1 * y + d
            a1 = a1 * y

            xlam = y

            x = x + 2.0d0 * pi

            y = q * DCOS(x / 3.0d0) - r
            a2 = a * y / 4.0d0 + b / 3.0d0
            a2 = a2 * y + c / 2.0d0
            a2 = a2 * y + d
            a2 = a2 * y

            IF ( a2 > a1 ) THEN

              xlam = y
              a1 = a2

            ENDIF

            x = x + 2.0d0 * pi

            y = q * DCOS(x / 3.0d0) - r
            a2 = a * y / 4.0d0 + b / 3.0d0
            a2 = a2 * y + c / 2.0d0
            a2 = a2 * y + d
            a2 = a2 * y

            IF ( a2 > a1 ) xlam = y

          ENDIF

          !=== New Phonon Functional Energy ===!
          !En = Eo - A/4 X**4 - B/3 X**3 - C/2 X**2 - D X!

          x = a * xlam / 4.0d0 + b / 3.0d0
          x = x * xlam + c / 2.0d0
          x = x * xlam + d
          x = x * xlam

          e = e - x

          !=== Output Data ===!

          IF ( myproc == 0 ) THEN

            CALL OUTPUTVIBE (a,b,c,d,e,df,xlam,istep)

          ENDIF

          !=== Calculate new |u> and (D-WMAX)|u> ===!

          DO i=1,nev
          DO j=1,nmat

            u(j,i) = u(j,i) + xlam * p(j,i)
            du(j,i) = du(j,i) + xlam * dp(j,i)

          ENDDO
          ENDDO

          !=== Calculate new <u|u> and <u|D|u> overlaps ===!

          DO i=1,nev
          DO j=i,nev

            x = 2.0d0 * spu(i,j) + xlam * spp(i,j)
            y = 2.0d0 * pdu(i,j) + xlam * pdp(i,j)

            suu(i,j) = suu(i,j) + x * xlam
            udu(i,j) = udu(i,j) + y * xlam

            suu(j,i) = suu(i,j)
            udu(j,i) = udu(i,j)

          ENDDO
          ENDDO

          !=== Write Backup File ===!

          m = 1

          IF ( nsbk /= 0 ) m = MODULO(istep,nsbk)
          IF ( istep == nsteps ) m = 0

          IF ( m == 0 .and. myproc == 0 ) THEN

            WRITE(10,*)nmat,nev

            DO i=1,nev

              fmat = '(2E16.8)'
              WRITE(10,fmat)0.0d0,0.0d0

              fmat = '(3E24.16)'
              WRITE(10,fmat)(u(j,i),j=1,nmat)

            ENDDO

            IF ( isym == 1 ) THEN

              fmat = '(3I6)'
              WRITE(10,fmat)nsite,nbas,nrow

              fmat = '(3E24.16)'

              DO k=1,nrow
              DO j=1,nbas

                WRITE(10,fmat)(gbas(i,j,k),i=1,3*nsite)

              ENDDO
              ENDDO

            ENDIF

            CLOSE (UNIT = 10)

            OPEN (UNIT = 10, FILE = rstfile, STATUS = 'Old')

          ENDIF

        ENDDO


        !=== Solve H|yi> = ai * S|yi> ===!

        ALLOCATE (wrk1(nev),wrk2(3*nev))

        CALL DSYGV (1,'V','U',nev,udu,nev,suu,nev,&
                  & wrk1,wrk2,3*nev,ierr)

        DEALLOCATE (wrk1,wrk2)

        IF ( ierr /= 0 ) THEN

          IF ( myproc == 0 ) THEN
          WRITE(4,*)'ERROR: DSYGV in VIBES failed.'
          WRITE(4,*)'DSYGV returned error code ierr = ',ierr
          ENDIF

          IF ( mpijob ) CALL CLOSEMPI

          STOP

        ENDIF

        DO i=1,nev
        DO j=1,nmat

          du(j,i) = 0.0d0

          DO k=1,nev
          du(j,i) = du(j,i) + u(j,k) * udu(k,i)
          ENDDO

        ENDDO
        ENDDO

        DO i=1,nev
        DO j=1,nmat
        u(j,i) = du(j,i)
        ENDDO
        ENDDO


        !=== Output Eigenvectors and Check Convergance ===!

 2      IF ( myproc == 0 ) THEN

          fmat = '(2A16)'

          WRITE(4,*)' '
          WRITE(4,fmat)'OMEGA (CM-1)    ','ERROR |DX-W^2 X|'
          WRITE(9,*)nmat,nev

        ENDIF

        DO i=1,nev

          IF ( isym == 0 ) THEN
          CALL DYNAMICALU (u(:,i),bk,sm)
          ELSE
          CALL GDYNAMICALU (u(:,i),bk,sm)
          ENDIF

          x = 0.0d0
          y = 0.0d0

          DO j=1,nmat
          x = x + bk(j) * u(j,i)
          y = y + bk(j) * bk(j)
          ENDDO

          y = y - x * x
          y = DSQRT(DABS(y))

          IF ( x < 0.0d0 ) THEN
            x = -cfac * DSQRT(DABS(x))
          ELSE
            x = cfac * DSQRT(x)
          ENDIF

          IF ( myproc == 0 ) THEN

            fmat = '(2E16.8)'
            WRITE(4,fmat)x,y
            WRITE(9,fmat)x,y

            fmat = '(3E24.16)'
            WRITE(9,fmat)(u(j,i),j=1,nmat)

          ENDIF

        ENDDO

        IF ( isym == 1 .and. myproc == 0 ) THEN

          fmat = '(3I6)'
          WRITE(9,fmat)nsite,nbas,nrow

          fmat = '(3E24.16)'

          DO k=1,nrow
          DO j=1,nbas

            WRITE(9,fmat)(gbas(i,j,k),i=1,3*nsite)

          ENDDO
          ENDDO

        ENDIF

        !=== Output Phonon Functional Info ===!

        IF ( myproc == 0 ) THEN

          WRITE(4,*)' '

          IF ( istep >= nsteps ) THEN

            WRITE(4,*)'WARNING: Maximum number of conjugate gradient'
            WRITE(4,*)'steps was reached in VIBES. Eigenvectors may'
            WRITE(4,*)'not have converged. NSTEPS = ',nsteps
            WRITE(4,*)' '

          ENDIF

          WRITE(4,*)'Final magnitude of Gradient = ',DSQRT(df)
          WRITE(4,*)'Final Phonon Functional Energy = ',e

        ENDIF

        DEALLOCATE (u,p,du,dp,g,bk,bkp,bkm)

        RETURN

      END SUBROUTINE VIBES
