! ==============================================================================
! Program: BFACTOR
! 
! Description: Computes the atomic Bfactors for a molecule from its low
!              frequency modes and outputs the information to a PDB file.
!
! Notes:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules -
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      PROGRAM BFACTOR

        IMPLICIT NONE

        INTEGER, DIMENSION(:), ALLOCATABLE :: itype

        DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: bfac,w
        DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE :: rat,u
        DOUBLE PRECISION, DIMENSION(:,:,:), ALLOCATABLE :: gbas

        INTEGER :: i,j,k,ib,ir,isym,nat,nev,nmat
        INTEGER :: istart,iadd,nsite,nbas,nrow

        DOUBLE PRECISION :: x(3),amass(106),c,xsq,err

        CHARACTER (LEN=70) :: gf2kfile,vibfile,outfile
        CHARACTER (LEN=2) :: atmn(106)

        DOUBLE PRECISION, PARAMETER :: fac = 5.510561825d5
        DOUBLE PRECISION, PARAMETER :: bmult = 100.0d0


        !=== Initialize Names ===!

        DATA atmn / ' H','HE','LI','BE',' B',' C',' N',' O',' F','NE', &
        & 'NA','MG','AL','SI',' P',' S','CL','AR',' K','CA','SC','TI', &
        & ' V','CR','MN','FE','CO','NI','CU','ZN','GA','GE','AS','SE', &
        & 'BR','KR','RB','SR',' Y','ZR','NB','MO','TC','RU','RH','PD', &
        & 'AG','CD','IN','SN','SB','TE',' I','XE','CS','BA','LA','CE', &
        & 'PR','ND','PM','SM','EU','GD','TB','DY','HO','ER','TM','TB', &
        & 'LU','HF','TA',' W','RE','OS','IR','PT','AU','HG','TL','PB', &
        & 'BI','PO','AT','RN','XX','XX','XX','XX','XX','XX','XX','XX', &
        & 'XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX'/

        iadd = 0

        amass(:) = 0.0d0

        amass(1) = 1.008d0
        amass(6) = 1.201d1
        amass(7) = 1.400d1
        amass(8) = 1.599d1
        amass(15)= 3.097d1
        amass(16)= 3.207d1

        !=== WELCOME ===!

        WRITE(*,*)'--------------- WELCOME TO BFACTOR -----------------'
        WRITE(*,*)' '
        WRITE(*,*)'This program will calculate the temperature factors '
        WRITE(*,*)'or (B Factors) for each atom in the molecule from   '
        WRITE(*,*)'the vibrational data and writes the information in a'
        WRITE(*,*)'PDB file.'
        WRITE(*,*)' '

        WRITE(*,*)'Enter the name of the input GF2K file.'
        READ(*,*)gf2kfile
        WRITE(*,*)' '

        WRITE(*,*)'Enter the name of the Saguaro vibrational file.'
        READ(*,*)vibfile
        WRITE(*,*)' '

        WRITE(*,*)'Enter the name of the output PDB file.'
        READ(*,*)outfile
        WRITE(*,*)' '

        WRITE(*,*)'Are the eigenvectors in symmetry coordinates?'
        WRITE(*,*)'Yes --- ENTER 1'
        WRITE(*,*)'No  --- ENTER 2'
        READ(*,*)isym
        WRITE(*,*)' '

        IF ( isym < 1 .or. isym > 2 ) THEN

          WRITE(*,*)'ERROR: Invalid choice. You entered ',isym
          STOP

        ENDIF

        OPEN (UNIT = 1, FILE = gf2kfile, STATUS = 'Unknown')
        OPEN (UNIT = 2, FILE = vibfile, STATUS = 'Unknown')
        OPEN (UNIT = 3, FILE = outfile, STATUS = 'Unknown')

        READ(1,*)nat
        READ(2,*)nmat,nev

        REWIND (UNIT = 2)

        ALLOCATE (itype(nat),rat(3,nat),bfac(nat))
        ALLOCATE (w(nev),u(nmat,nev))

        READ(1,*)(itype(i),rat(1,i),rat(2,i),rat(3,i),i=1,nat)


        !=== Calculate B Factors ===!

        bfac(:) = 0.0d0

 1      IF ( iadd == 1 ) THEN

          WRITE(*,*)'Enter the name of the Saguaro vibrational file.'
          READ(*,*)vibfile
          WRITE(*,*)' '

          OPEN (UNIT = 2, FILE = vibfile, STATUS = 'Unknown')

        ENDIF

        READ(2,*)nmat,nev

        DO i=1,nev

          READ(2,*)w(i),err
          READ(2,20)(u(j,i),j=1,nmat)

        ENDDO

        IF ( isym == 1 ) THEN

          READ(2,*)nsite,nbas,nrow

          ALLOCATE (gbas(3,nbas,nrow))

          DO ir=1,nrow
          DO ib=1,nbas

            READ(2,20)gbas(1,ib,ir),gbas(2,ib,ir),gbas(3,ib,ir)

            IF ( nsite > 1 ) THEN

              DO i=2,nsite
              READ(2,20)x(1),x(2),x(3)
              ENDDO

            ENDIF

          ENDDO
          ENDDO

        ENDIF

        WRITE(*,*)'Enter the starting eigenvector - DEFAULT = 7'
        READ(*,*)istart
        WRITE(*,*)' '

        DO i=istart,nev
        DO j=1,nat

          IF ( isym == 1 ) THEN

            k = nbas * ( j - 1 )

            DO ir=1,nrow

              x(:) = 0.0d0

              DO ib=1,nbas
              x(:) = x(:) + gbas(:,ib,ir) * u(k+ib,i)
              ENDDO

              xsq = x(1) * x(1) + x(2) * x(2) + x(3) * x(3)
              c = w(i) * w(i)
              c = xsq / c

              bfac(j) = bfac(j) + c

            ENDDO

          ELSE

            k = 3 * ( j - 1 )

            x(1) = u(k+1,i)
            x(2) = u(k+2,i)
            x(3) = u(k+3,i)

            xsq = x(1) * x(1) + x(2) * x(2) + x(3) * x(3)
            c = w(i) * w(i)
            c = xsq / c

            bfac(j) = bfac(j) + c

          ENDIF

        ENDDO
        ENDDO

        CLOSE (UNIT = 2)

        WRITE(*,*)'Do you have another vibrational file to add?'
        WRITE(*,*)'Yes --- ENTER 1'
        WRITE(*,*)'No  --- ENTER 2'
        READ(*,*)iadd
        WRITE(*,*)' '

        IF ( iadd == 1 ) GOTO 1


        !=== Multiply by B Factor by Conversion ===!

        DO i=1,nat

          j = itype(i)

          IF ( amass(j) == 0.0d0 ) THEN

            WRITE(*,*)'ERROR: There is no mass for atom type.'
            WRITE(*,*)'       itype = ',itype(i)
            STOP

          ENDIF

          c = fac / amass(j)

          bfac(i) = bfac(i) * c * bmult

        ENDDO

        !=== Write out PDB File ===!

        DO i=1,nat

          j = itype(i)

          WRITE(3,30)'ATOM  ',i,atmn(j),'UNK','A',1,rat(1,i),&
                      & rat(2,i),rat(3,i),1.0d0,bfac(i),j

        ENDDO

        WRITE(3,31)'TER   ',nat+1,'UNK','A',1

        WRITE(3,32)'END'

        CLOSE (UNIT=1)
        CLOSE (UNIT=3)

        !=== DEALLOCATE ARRAYS ===!

        DEALLOCATE (itype,rat,w,u,bfac)
        IF ( isym == 1 ) DEALLOCATE (gbas)

        !=== Format Statements ===!

 20     FORMAT (3E24.16)
 30     FORMAT (A6,I5,1X,A2,3X,A3,1X,A1,I4,4X,3F8.3,2F6.2,I4)
 31     FORMAT (A6,I5,6X,A3,1X,A1,I4)
 32     FORMAT (A3)

        END PROGRAM BFACTOR
