! ==============================================================================
! Program: DVECTOR
! 
! Description: Constructs an XYZ coordinate file with mode eigenvector
!              from a Saguaro Vibrational data file.
!
! Notes:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules -
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      PROGRAM DVECTOR

        IMPLICIT NONE

        !=== VARIABLES ===!

        DOUBLE PRECISION, DIMENSION(:,:,:), ALLOCATABLE :: grot,gbas
        DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE :: rat,r,evec,rvec
        DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: w

        INTEGER, DIMENSION(:), ALLOCATABLE :: itype,lca

        DOUBLE PRECISION :: sm(106),v(3),amp,time,err,xv,c1

        INTEGER :: i,j,k,jj,kk,nat,iev,isym,iout,is,ir,n
        INTEGER :: ngrp,nsite,nrow,nbas,nmat,nev,nca,ica,iatm

        CHARACTER :: hash
        CHARACTER (LEN=2) :: atmn(106)
        CHARACTER (LEN=70) :: vibfile,gf2kfile,ramfile,xyzfile,symfile


        nrow = 1
        nsite = 1

        !=== Initialize Names ===!

        DATA atmn / ' H','He','Li','Be',' B',' C',' N',' O',' F','Ne', &
        & 'Na','Mg','Al','Si',' P',' S','Cl','Ar',' K','Ca','Sc','Ti', &
        & ' V','Cr','Mn','Fe','Co','Ni','Cu','Zn','Ga','Ge','As','Se', &
        & 'Br','Kr','Rb','Sr',' Y','Zr','Nb','Mo','Tc','Ru','Rh','Pd', &
        & 'Ag','Cd','In','Sn','Sb','Te',' I','Xe','Cs','Ba','La','Ce', &
        & 'Pr','Nd','Pm','Sm','Eu','Gd','Tb','Dy','Ho','Er','Tm','Tb', &
        & 'Lu','Hf','Ta',' W','Re','Os','Ir','Pt','Au','Hg','Tl','Pb', &
        & 'Bi','Po','At','Rn','XX','XX','XX','XX','XX','XX','XX','XX', &
        & 'XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX'/

        !=== Initialize Inv Sqrt of Masses ===!

        sm(:) = 0.0d0

        sm(1) = 1.0d0 / DSQRT(1.008d0)
        sm(3) = 1.0d0 / DSQRT(6.941d0)
        sm(6) = 1.0d0 / DSQRT(1.201d1)
        sm(7) = 1.0d0 / DSQRT(1.401d1)
        sm(8) = 1.0d0 / DSQRT(1.599d1)
        sm(9) = 1.0d0 / DSQRT(1.899d1)
        sm(11)= 1.0d0 / DSQRT(2.299d1)
        sm(12)= 1.0d0 / DSQRT(2.431d1)
        sm(13)= 1.0d0 / DSQRT(2.698d1)
        sm(14)= 1.0d0 / DSQRT(2.809d1)
        sm(15)= 1.0d0 / DSQRT(3.097d1)
        sm(16)= 1.0d0 / DSQRT(3.207d1)
        sm(19)= 1.0d0 / DSQRT(3.909d1)
        sm(20)= 1.0d0 / DSQRT(4.078d1)
        sm(29)= 1.0d0 / DSQRT(6.355d1)
        sm(32)= 1.0d0 / DSQRT(7.261d1)
        sm(79)= 1.0d0 / DSQRT(1.969d2)

        !=== WELCOME ===!

        WRITE(*,*)'---------------- WELCOME TO DVECTOR ----------------'
        WRITE(*,*)' '
        WRITE(*,*)'This program will construct an XYZ file containing  '
        WRITE(*,*)'the displacement vector for a vibrational mode.'
        WRITE(*,*)' '

        WRITE(*,*)'Enter the name of the Saguaro vibrational file.'
        READ(*,*)vibfile
        WRITE(*,*)' '

        WRITE(*,*)'Enter the name of the vibrational profile file.'
        WRITE(*,*)'NOTE: This file is computed by VIBPROFILE.'
        READ(*,*)ramfile
        WRITE(*,*)' '

        WRITE(*,*)'Enter the name of the GF2K file.'
        READ(*,*)gf2kfile
        WRITE(*,*)' '

        WRITE(*,*)'Enter the name of the output XYZ file.'
        READ(*,*)xyzfile
        WRITE(*,*)' '

        WRITE(*,*)'Are the eigenvectors in symmetry coordinates?'
        WRITE(*,*)'Yes --- ENTER 1'
        WRITE(*,*)'No  --- ENTER 2'
        READ(*,*)isym
        WRITE(*,*)' '

        IF ( isym < 1 .or. isym > 2 ) THEN

          WRITE(*,*)'ERROR: Invalid choice. You entered ',isym
          STOP

        ENDIF

        IF ( isym == 1 ) THEN

          WRITE(*,*)'Enter the name of the symmetry file.'
          READ(*,*)symfile
          WRITE(*,*)' '

          OPEN (UNIT = 1, FILE = symfile, STATUS = 'Unknown')

          READ(1,*)ngrp

          ALLOCATE (grot(3,3,ngrp))

          DO i=1,ngrp
          READ(2,*)grot(1,1,i),grot(1,2,i),grot(1,3,i)
          READ(2,*)grot(2,1,i),grot(2,2,i),grot(2,3,i)
          READ(2,*)grot(3,1,i),grot(3,2,i),grot(3,3,i)
          ENDDO

          CLOSE (UNIT = 1)

        ENDIF

        !=== User Options ===!

        WRITE(*,*)'What atoms do you want to show in the Disp. Vector?'
        WRITE(*,*)'ALL Atoms                         --- ENTER 1'
        WRITE(*,*)'Only peptide CA atoms             --- ENTER 2'
        READ(*,*)iout
        WRITE(*,*)' '

        amp = 1.0d0

        WRITE(*,*)'Enter amplitude to apply to the displacement vector.'
        WRITE(*,*)'NOTE: Suggested for large structures 100.00'
        READ(*,*)amp
        WRITE(*,*)' '

        OPEN (UNIT=1, FILE=vibfile, STATUS='Unknown')
        OPEN (UNIT=2, FILE=gf2kfile,STATUS='Unknown')
        OPEN (UNIT=3, FILE=ramfile, STATUS='Unknown')
        OPEN (UNIT=4, FILE=xyzfile, STATUS='Unknown')


        !=== Read in Data ===!

        !=== Read in Vibrational Data ===!

        READ(1,*)nmat,nev

        ALLOCATE (evec(nmat,nev),w(nev))

        DO iev=1,nev

          READ(1,*)w(iev)

          READ(1,10)(evec(i,iev),i=1,nmat)

        ENDDO

        !=== Read in Basis Vectors ===!

        IF ( isym == 1 ) THEN

          READ(1,*)nsite,nbas,nrow

          ALLOCATE (gbas(3*nsite,nbas,nrow))

          DO k=1,nrow
          DO j=1,nbas
          READ(1,10)(gbas(i,j,k),i=1,3*nsite)
          ENDDO
          ENDDO

        ENDIF

        !=== Read in Coordinates ===!

        READ(2,*)nat,time

        ALLOCATE (rat(3,nat),itype(nat))
        ALLOCATE (r(3,nat),rvec(3,nat),lca(nat))

        READ(2,*)(itype(i),rat(1,i),rat(2,i),rat(3,i),i=1,nat)

        !=== Form List of Calpha ===!

        lca(:) = 0
        nca = 0

        DO i=3,nat
        IF ( itype(i) == 7 ) THEN
        IF ( itype(i-2) == 6 .and. itype(i-1) == 8 ) THEN

          IF ( itype(i+1) == 1 .and. itype(i+2) == 6 ) THEN

            nca = nca + 1
            lca(i) = -1
            lca(i+2) = 1

          ELSEIF ( itype(i+1) == 6 .and. itype(i+2) == 1 ) THEN

            nca = nca + 1
            lca(i) = -1
            lca(i+1) = 1

          ENDIF

        ELSEIF ( itype(i-2) == 8 .and. itype(i-1) == 8 ) THEN

          IF ( itype(i+1) == 1 .and. itype(i+2) == 1 ) THEN
          IF ( itype(i+3) == 6 ) THEN

            nca = nca + 1
            lca(i) = -1
            lca(i+3) = 1

          ELSE

            nca = nca + 1
            lca(i) = -1
            lca(i+4) = 1

          ENDIF
          ENDIF

        ENDIF
        ENDIF
        ENDDO

        !=== Find number of atoms to write out ===!

        IF ( iout == 1 ) n = nat * nsite
        IF ( iout == 2 ) n = nca * nsite


        !=== CREATE DISPLACEMENT PATTERNS ===!

        READ(3,*)hash

        DO iev=1,nev
        DO ir=1,nrow

        READ(3,*)hash

        IF ( hash == 'X' .or. hash == 'x' ) THEN

          DO is=1,nsite

            !=== Get coordinates ===!

            IF ( isym == 1 ) THEN

              r(:,:) = 0.0d0

              DO i=1,nat
              DO j=1,3
              r(:,i) = r(:,i) + grot(:,j,is) * rat(j,i)
              ENDDO
              ENDDO

            ELSE

              r(:,:) = rat(:,:)

            ENDIF

            !=== Form Displacement ===!

            DO i=1,nat

              c1 = sm(itype(i))

              IF ( isym == 1 ) THEN

                jj = nbas * ( i - 1 )
                kk = 3 * ( is - 1 )

                rvec(:,i) = 0.0d0

                DO j=1,nbas
                DO k=1,3
                rvec(k,i) = rvec(k,i) + c1 * &
                          & gbas(kk+k,j,ir) * evec(jj+j,iev)
                ENDDO
                ENDDO

              ELSE

                jj = 3 * ( i - 1 )

                DO j=1,3
                rvec(j,i) = c1 * evec(jj+j,iev)
                ENDDO

              ENDIF

            ENDDO

            !=== Fix Displacement Vector ===!

            IF ( iout == 2 ) THEN

              v(:) = 0.0d0
              ica = 0
              iatm = 0

              DO i=1,nat

                IF ( lca(i) == -1 .and. ica /= 0 ) THEN

                  rvec(:,ica) = v(:) / DBLE(iatm)

                  ica = 0
                  iatm = 0
                  v(:) = 0.0d0

                ELSEIF ( i == nat .and. ica /= 0 ) THEN

                  rvec(:,ica) = v(:) / DBLE(iatm)

                ELSE

                  v(:) = v(:) + rvec(:,i)

                  rvec(:,i) = 0.0d0

                  iatm = iatm + 1

                  IF ( ica == 0 .and. lca(i) == 1 ) ica = i

                ENDIF

              ENDDO

            ENDIF

            !=== Normalize Vector ===!

            xv = 0.0d0
            DO i=1,nat
            xv = xv + rvec(1,i)**2 + rvec(2,i)**2 + rvec(3,i)**2
            ENDDO

            xv = amp / DSQRT(xv)

            rvec(:,:) = xv * rvec(:,:)

            !=== Output XYZ File ===!

            IF ( is == 1 ) THEN

              WRITE(4,40)n
              WRITE(4,41)'OMEGA (CM-1) = ',w(iev)

            ENDIF

            IF ( iout == 2 ) THEN

              DO i=1,nat
              IF ( lca(i) == 1 ) THEN
              WRITE(4,42)atmn(itype(i)),r(1,i),r(2,i),r(3,i),&
                       & rvec(1,i),rvec(2,i),rvec(3,i)
              ENDIF
              ENDDO

            ELSE

              WRITE(4,42)(atmn(itype(i)),r(1,i),r(2,i),r(3,i),&
                       & rvec(1,i),rvec(2,i),rvec(3,i),i=1,nat)

            ENDIF

          ENDDO

        ENDIF

        ENDDO
        ENDDO

        WRITE(*,*)'The displacement vectors have been formed. Goodbye.'

        CLOSE (UNIT = 1)
        CLOSE (UNIT = 2)
        CLOSE (UNIT = 3)
        CLOSE (UNIT = 4)

        !=== DEALLOCATE ARRAYS ===!

        DEALLOCATE (rat,r,evec,rvec,w,lca,itype)
        IF ( isym == 1 ) DEALLOCATE (gbas,grot)

        !=== Format Statements ===!

 10     FORMAT(3E24.16)
 40     FORMAT(I8)
 41     FORMAT(A15,E16.8)
 42     FORMAT(A2,6F13.7)

      END PROGRAM DVECTOR
