! ==============================================================================
! Program: TRAJ2XYZ
! 
! Description: Converts a Saguaro trajecory file to a file containing
!              multiple snapshots in XYZ format concatenated together.
!
! Notes:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules -
! Functions -
! Subroutines - GETRLV
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      PROGRAM TRAJ2XYZ

        IMPLICIT NONE

        !=== VARIABLES ===!

        DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE :: rat,rmol
        INTEGER, DIMENSION(:), ALLOCATABLE :: itype,natmol,moln
        INTEGER, DIMENSION(:), ALLOCATABLE :: n1,n2,n3
        CHARACTER, DIMENSION(:), ALLOCATABLE :: mtype

        DOUBLE PRECISION :: time,c,a1(3),a2(3),a3(3)
        DOUBLE PRECISION :: x(3),b1(3),b2(3),b3(3)

        CHARACTER (LEN=2) :: aname,atmn(106)
        CHARACTER (LEN=70) :: gf2kfile,xyzfile,molfile

        INTEGER :: i,j,ifrm,ipbc,iout,irm,nat,natm,nmol
        INTEGER, PARAMETER :: maxfrm = 100000


        !=== Initialize Names ===!

        DATA atmn / ' H','He','Li','Be',' B',' C',' N',' O',' F','Ne', &
        & 'Na','Mg','Al','Si',' P',' S','Cl','Ar',' K','Ca','Sc','Ti', &
        & ' V','Cr','Mn','Fe','Co','Ni','Cu','Zn','Ga','Ge','As','Se', &
        & 'Br','Kr','Rb','Sr',' Y','Zr','Nb','Mo','Tc','Ru','Rh','Pd', &
        & 'Ag','Cd','In','Sn','Sb','Te',' I','Xe','Cs','Ba','La','Ce', &
        & 'Pr','Nd','Pm','Sm','Eu','Gd','Tb','Dy','Ho','Er','Tm','Tb', &
        & 'Lu','Hf','Ta',' W','Re','Os','Ir','Pt','Au','Hg','Tl','Pb', &
        & 'Bi','Po','At','Rn','XX','XX','XX','XX','XX','XX','XX','XX', &
        & 'XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX','XX'/

        !=== WELCOME ===!

        WRITE(*,*)'-------------- WELCOME TO TRAJ2XYZ -----------------'
        WRITE(*,*)' '
        WRITE(*,*)'This program converts a Saguaro MD trajectory file'
        WRITE(*,*)'with multiple snapshots into a concat. XYZ file.'
        WRITE(*,*)' '

        WRITE(*,*)'Enter the name of the input trajectory file.'
        READ(*,*)gf2kfile
        WRITE(*,*)' '

        WRITE(*,*)'Enter the name of the output XYZ file.'
        READ(*,*)xyzfile
        WRITE(*,*)' '


        !=== Get User Options ===!

        WRITE(*,*)'Do you have periodic boundary conditions?'
        WRITE(*,*)'YES --- ENTER 1'
        WRITE(*,*)'NO  --- ENTER 2'
        READ(*,*)ipbc
        WRITE(*,*)' '

        IF ( ipbc < 1 .or. ipbc > 2 ) THEN

          WRITE(*,*)'ERROR: Invalid choice. You entered ',ipbc
          STOP

        ENDIF

        IF ( ipbc == 1 ) THEN

          WRITE(*,*)'Would you like to:'
          WRITE(*,*)'Keep molecules where they are?      ---- ENTER 1'
          WRITE(*,*)'Move molecules back to unit cell?   ---- ENTER 2'
          READ(*,*)iout
          WRITE(*,*)' '

          IF ( iout < 1 .or. iout > 2 ) THEN

            WRITE(*,*)'ERROR: Invalid choice. You entered ',iout
            STOP

          ENDIF

        ELSE

          iout = 1

        ENDIF

        WRITE(*,*)'What molecules should be output?'
        WRITE(*,*)'All molecules          ---- ENTER 1'
        WRITE(*,*)'Only solute molecules  ---- ENTER 2'
        READ(*,*)irm
        WRITE(*,*)' '

        IF ( irm < 1 .or. irm > 2 ) THEN

          WRITE(*,*)'ERROR: Invalid choice. You entered ',irm
          STOP

        ENDIF


        !=== SETUP ===!

        !=== Open Files ===!

        OPEN (UNIT=1,FILE=gf2kfile,STATUS='Unknown')
        OPEN (UNIT=2,FILE=xyzfile,STATUS='Unknown')

        !=== Allocate Arrays ===!

        READ(1,*)nat,time

        REWIND (UNIT = 1)

        ALLOCATE(itype(nat),rat(3,nat))

        !=== Read In MOL file if Required ===!

        IF ( irm == 2 .or. iout == 2 ) THEN

          WRITE(*,*)'Enter the name of the Saguaro MOL file.'
          READ(*,*)molfile

          OPEN (UNIT=3,FILE=molefile,STATUS='Unknown')

          READ(3,*)j,nmol

          IF ( j /= nat ) THEN

            WRITE(*,*)'ERROR: Number of atoms in MOL file does not'
            WRITE(*,*)'       the coordinate file.'
            STOP

          ENDIF

          ALLOCATE (moln(nat),mtype(nat))

          READ(3,*)(moln(i),mtype(i),i=1,nat)

          CLOSE (UNIT = 3)

        ENDIF

        !=== Count Number of atoms per molecule ===!

        IF ( iout == 2 ) THEN

          ALLOCATE (rmol(3,nmol),natmol(nmol))
          ALLOCATE (n1(nmol),n2(nmol),n3(nmol))

          natmol(:) = 0

          DO i=1,nat

            j = moln(i)
            natmol(j) = natmol(j) + 1

          ENDDO

        ENDIF

        !=== Count non-solvent atoms ===!

        natm = nat

        IF ( irm == 2 ) THEN

          natm = 0

          DO i=1,nat
          IF ( mtype(i) /= 'S' ) natm = natm + 1
          ENDIF

        ENDIF


        !=== Write out Trajectory in XYZ Format ===!

        DO ifrm=1,maxfrm

          !=== Read in Trajectory Snapshot ===!

          READ(1,*,END=10)nat,time

          READ(1,*)(itype(i),rat(1,i),rat(2,i),rat(3,i),i=1,nat)

          IF ( ipbc == 1 ) THEN

            READ(1,*)a1(1),a2(1),a3(1)
            READ(1,*)a1(2),a2(2),a3(2)
            READ(1,*)a1(3),a2(3),a3(3)

            CALL GETRLV (a1,a2,a3,b1,b2,b3)

          ENDDO

          !=== Translate molecules back to unit cell ===!

          IF ( iout == 2 ) THEN

            rmol(:,:) = 0.0d0

            DO i=1,nat

              j = moln(i)

              c = 1.0d0 / DBLE(natmol(j))

              rmol(:,j) = rmol(:,j) + c * rat(:,i)

            ENDDO

            DO i=1,nmol

              x(:) = 0.0d0

              DO j=1,3
              x(1) = x(1) + b1(j) * rmol(j,i)
              x(2) = x(2) + b2(j) * rmol(j,i)
              x(3) = x(3) + b3(j) * rmol(j,i)
              ENDDO

              n1(i) = INT(x(1))
              n2(i) = INT(x(2))
              n3(i) = INT(x(3))

              IF ( x(1) < 0 ) n1(i) = n1(i) - 1
              IF ( x(2) < 0 ) n2(i) = n2(i) - 1
              IF ( x(3) < 0 ) n3(i) = n3(i) - 1

            ENDDO

            DO i=1,nat

              j = moln(i)

              rat(:,i) = rat(:,i) - a1(:) * n1(j) &
                     & - a2(:) * n2(j) - a3(:) * n3(j)

            ENDDO

          ENDIF

          !=== Output XYZ File ===!

          WRITE(2,20)natm

          WRITE(2,21)'TIME = ',time

          DO i=1,nat

            aname = atmn(itype(i))

            IF ( irm == 1 .or. mtype(i) /= 'S' ) THEN

              WRITE(2,22)aname,rat(1,i),rat(2,i),rat(3,i)

            ENDIF

          ENDDO

        ENDDO

        WRITE(*,*)'ERROR: The trajectory file has more snapshots.'
        WRITE(*,*)'       Increase MAXFRM.'
        STOP

 10     WRITE(*,*)'The trajectory file has been converted. Goodbye.'

        CLOSE (UNIT = 1)
        CLOSE (UNIT = 2)

        !=== DEALLOCATE ARRAYS ===!

        DEALLOCATE (rat,itype)
        IF ( irm == 2 .or. iout == 2 ) DEALLOCATE (moln,mtype)
        IF ( iout == 2 ) DEALLOCATE (rmol,natmol,n1,n2,n3)

 20     FORMAT(I8)
 21     FORMAT(A7,E16.8)
 22     FORMAT(A2,3F13.7)

      END PROGRAM TRAJ2XYZ

! ==============================================================================
! Subroutine: GETRLV (A1,A2,A3,B1,B2,B3)
! 
! Purpose: Given lattice vectors A1,A2, and A3, computes the reciprocal
!          lattice vectors B1,B2, and B3.
!
! Method: The Reciprocal lattice vectors are defined as:
!
!            b1 = ( a2 X a3 ) / V
!            b2 = ( a3 X a1 ) / V
!            b3 = ( a1 X a2 ) / V
!
!            WITH -- V = a1 . ( a2 X a3 )
!
! Arguments:
!
!           A1 - The first direct lattice vector. (INPUT)
!           A2 - The second direct lattice vector. (INPUT)
!           A3 - The third direct lattice vector. (INPUT)
!           B1 - The first reciprocal lattice vector. (OUTPUT)
!           B2 - The second reciprocal lattice vector. (OUTPUT)
!           B3 - The third reciprocal lattice vector. (OUTPUT)
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules -
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE GETRLV (A1,A2,A3,B1,B2,B3)

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(IN) :: a1(3),a2(3),a3(3)
        DOUBLE PRECISION, INTENT(OUT) :: b1(3),b2(3),b3(3)

        !=== VARIABLES ===!

        DOUBLE PRECISION :: x


        !=== First R Lattice Vector ===!

        b1(1) = a2(2) * a3(3) - a2(3) * a3(2)
        b1(2) = a2(3) * a3(1) - a2(1) * a3(3)
        b1(3) = a2(1) * a3(2) - a2(2) * a3(1)

        !=== Second R Lattice Vector ===!

        b2(1) = a3(2) * a1(3) - a3(3) * a1(2)
        b2(2) = a3(3) * a1(1) - a3(1) * a1(3)
        b2(3) = a3(1) * a1(2) - a3(2) * a1(1)

        !=== Third R Lattice Vector ===!

        b3(1) = a1(2) * a2(3) - a1(3) * a2(2)
        b3(2) = a1(3) * a2(1) - a1(1) * a2(3)
        b3(3) = a1(1) * a2(2) - a1(2) * a2(1)

        x = a1(1) * b1(1) + a1(2) * b1(2) + a1(3) * b1(3)

        b1(:) = b1(:) / x
        b2(:) = b2(:) / x
        b3(:) = b3(:) / x

        RETURN

      END SUBROUTINE GETRLV
