! ==============================================================================
! Program: VIBPROFILE
! 
! Description: Performs an analysis of the vibrational modes computed in
!              Saguaro. The program calculates the following:
!              (1) The contribution to the Raman Scattering Intensity
!                  from the mode. (See Dykeman et al. Phys. Rev. E)
!              (2) The participation number of the mode which gives
!                  an estimate of the number of atoms moving.
!
! Notes:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules -
! Functions -
! Subroutines - NEIGHBORS
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      PROGRAM VIBPROFILE

        IMPLICIT NONE

        !=== VARIABLES ===!

        DOUBLE PRECISION, DIMENSION(:,:,:), ALLOCATABLE :: grot,gbas
        DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE :: rat,evec
        DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: prob,w

        INTEGER, DIMENSION(:,:), ALLOCATABLE :: lbd,lneigh
        INTEGER, DIMENSION(:), ALLOCATABLE :: itype,nneigh

        DOUBLE PRECISION :: p(3,3),dp(3,3),chi(3,3),sm(106),parm(3)
        DOUBLE PRECISION :: v(3),r12(3),r12e(3),edf(3),c1,c2,c3,x,d
        DOUBLE PRECISION :: time,err,stot,wmult,theta,rint,rlam,rmu

        INTEGER :: i,j,k,ii,jj,kk,iat,jat,iev,is,ir,isym
        INTEGER :: nsite,nbas,nrow,ngrp,nat,nbd,nev,nmat

        CHARACTER (LEN=70) :: vibfile,gf2kfile,outfile,symfile

        DOUBLE PRECISION, PARAMETER :: small = 1.0d-10
        INTEGER, PARAMETER :: nnmax = 4


        nsite = 1
        nrow = 1

        !=== PARAMETER DATA ===!

        !SNOKE AND CARDONA!

        parm(1) = 0.50d0    !(ALPHA_PAR - ALPHA_PER)!
        parm(2) = 1.37d0    !D(ALPHA_PAR)/DX!
        parm(3) = 0.17d0    !D(ALPHA_PER)/DX!

        !MENENDEZ, PAGE, ADAMS!

        parm(1) = 0.64d0    !(ALPHA_PAR - ALPHA_PER)!
        parm(2) = 1.15d0    !D(ALPHA_PAR)/DX!
        parm(3) = 0.00d0    !D(ALPHA_PER)/DX!

        !=== Initialize Inv Sqrt of Masses ===!

        sm(:) = 0.0d0

        sm(1) = 1.0d0 / DSQRT(1.008d0)
        sm(3) = 1.0d0 / DSQRT(6.941d0)
        sm(6) = 1.0d0 / DSQRT(1.201d1)
        sm(7) = 1.0d0 / DSQRT(1.401d1)
        sm(8) = 1.0d0 / DSQRT(1.599d1)
        sm(9) = 1.0d0 / DSQRT(1.899d1)
        sm(11)= 1.0d0 / DSQRT(2.299d1)
        sm(12)= 1.0d0 / DSQRT(2.431d1)
        sm(13)= 1.0d0 / DSQRT(2.698d1)
        sm(14)= 1.0d0 / DSQRT(2.809d1)
        sm(15)= 1.0d0 / DSQRT(3.097d1)
        sm(16)= 1.0d0 / DSQRT(3.207d1)
        sm(19)= 1.0d0 / DSQRT(3.909d1)
        sm(20)= 1.0d0 / DSQRT(4.078d1)
        sm(29)= 1.0d0 / DSQRT(6.355d1)
        sm(32)= 1.0d0 / DSQRT(7.261d1)
        sm(79)= 1.0d0 / DSQRT(1.969d2)

        !=== WELCOME ===!

        WRITE(*,*)'-------------- WELCOME TO VIBPROFILE ---------------'
        WRITE(*,*)' '
        WRITE(*,*)'This program will perform an analysis on a Saguaro  '
        WRITE(*,*)'vibrational data file.'
        WRITE(*,*)' '

        WRITE(*,*)'Enter the name of the Saguaro vibrational file.'
        READ(*,*)vibfile
        WRITE(*,*)' '

        WRITE(*,*)'Enter the name of the input GF2K file.'
        READ(*,*)gf2kfile
        WRITE(*,*)' '

        WRITE(*,*)'Enter the name of the output file.'
        READ(*,*)outfile
        WRITE(*,*)' '

        WRITE(*,*)'Are the eigenvectors in symmetry coordinates?'
        WRITE(*,*)'Yes --- ENTER 1'
        WRITE(*,*)'No  --- ENTER 2'
        READ(*,*)isym
        WRITE(*,*)' '

        IF ( isym < 1 .or. isym > 2 ) THEN

          WRITE(*,*)'ERROR: Invalid choice. You entered ',isym
          STOP

        ENDIF

        IF ( isym == 1 ) THEN

          WRITE(*,*)'Enter the name of the symmetry file.'
          READ(*,*)symfile
          WRITE(*,*)' '

          OPEN (UNIT = 1, FILE = symfile, STATUS = 'Unknown')

          READ(1,*)ngrp

          ALLOCATE (grot(3,3,ngrp))

          DO i=1,ngrp
          READ(2,*)grot(1,1,i),grot(1,2,i),grot(1,3,i)
          READ(2,*)grot(2,1,i),grot(2,2,i),grot(2,3,i)
          READ(2,*)grot(3,1,i),grot(3,2,i),grot(3,3,i)
          ENDDO

          CLOSE (UNIT = 1)

        ENDIF


        !=== Get User Options ===!

        WRITE(*,*)'Enter Raman scattering detector angle (deg).'
        READ(*,*)theta
        WRITE(*,*)' '

        WRITE(*,*)'Using the following polarization data:'
        WRITE(*,*)' '
        WRITE(*,*)'(ALPHA_PAR - ALPHA_PER) = ',PARM(1)
        WRITE(*,*)'D(ALPHA_PAR)/DX = ',PARM(2)
        WRITE(*,*)'D(ALPHA_PER)/DX = ',PARM(3)
        WRITE(*,*)' '

        theta = theta * DATAN(1.0d0) / 45.0d0

        x = DSIN(theta)

        rlam = 4.0d0 - 6.0d0 * x * x
        rmu = 14.0d0 - x * x

        OPEN (UNIT = 1, FILE = vibfile, STATUS = 'Unknown')
        OPEN (UNIT = 2, FILE = gf2kfile, STATUS = 'Unknown')
        OPEN (UNIT = 3, FILE = outfile, STATUS = 'Unknown')


        !=== Read in Data ===!

        !=== Read in vibrational data ===!

        READ(1,*)nmat,nev

        ALLOCATE (evec(nmat,nev),w(nev))

        DO iev=1,nev

          READ(1,*)w(iev)

          READ(1,10)(evec(i,iev),i=1,nmat)

        ENDDO

        !=== Read in Basis Vectors ===!

        IF ( isym == 1 ) THEN

          READ(1,*)nsite,nbas,nrow

          ALLOCATE (gbas(3*nsite,nbas,nrow))

          DO k=1,nrow
          DO j=1,nbas
          READ(1,10)(gbas(i,j,k),i=1,3*nsite)
          ENDDO
          ENDDO

        ENDIF

        !=== Read in Coordinates ===!

        READ(2,*)nat,time

        ALLOCATE (rat(3,nat),itype(nat),lneigh(nnmax,nat))
        ALLOCATE (nneigh(nat),prob(nat))

        READ(2,*)(itype(i),rat(1,i),rat(2,i),rat(3,i),i=1,nat)

        !=== Form Bond List ===!

        CALL NEIGHBORS (rat,itype,lneigh,nneigh,nat,nnmax)

        nbd = 0

        DO i=1,nat
        DO j=1,nneigh(i)
        IF ( lneigh(j,i) > i ) nbd = nbd + 1
        ENDDO
        ENDDO

        IF ( nbd == 0 ) THEN

          WRITE(*,*)'ERROR: No bonds found. Something is strange.'
          WRITE(*,*)'Check NEIGHBORS.'
          STOP

        ENDIF

        ALLOCATE (lbd(2,nbd))

        k = 0

        DO i=1,nat
        DO j=1,nneigh(i)

          iat = lneigh(j,i)

          IF ( iat > i ) THEN

            k = k + 1
            lbd(1,k) = i
            lbd(2,k) = iat

          ENDIF

        ENDDO
        ENDDO


        !=== Begin Analysis ===!

        WRITE(3,30)'#','    OMEGA (CM-1)','    MULTIPLICITY',&
                      &'      RAMAN INT.'

        DO iev=1,nev
        DO ir=1,nrow

          !=== Calculate Participation Number ===!

          prob(:) = 0.0d0

          DO i=1,nat
          DO is=1,nsite

            c1 = sm(itype(i))

            IF ( isym == 1 ) THEN

              jj = nbas * ( i - 1 )
              kk = 3 * ( is - 1 )

              v(:) = 0.0d0

              DO j=1,nbas
              DO k=1,3
              v(k) = v(k) + c1 * gbas(kk+k,j,ir) * evec(jj+j,iev)
              ENDDO
              ENDDO

            ELSE

              jj = 3 * ( i - 1 )

              DO j=1,3
              v(j) = c1 * evec(jj+j,iev)
              ENDDO

            ENDIF

            prob(i) = prob(i) + v(1)**2 + v(2)**2 + v(3)**2

          ENDDO
          ENDDO

          x = SUM(prob)

          x = 1.0d0 / x

          stot = 0.0d0

          DO i=1,nat

            prob(i) = x * prob(i)

            IF ( prob(i) > small ) THEN
            stot = stot - prob(i) * DLOG(prob(i))
            ENDIF

          ENDDO

          wmult = DEXP(stot)

          !=== Calculate Susceptability ===!

          edf(:) = 0.0d0
          chi(:,:) = 0.0d0

          DO i=1,nbd
          DO is=1,nsite

            iat = lbd(1,i)
            jat = lbd(2,i)

            !=== Construct R12 ===!

            r12(j) = rat(:,iat) - rat(:,jat)

            !=== Construct R12E ===!

            c1 = sm(itype(iat))
            c2 = sm(itype(jat))

            IF ( isym == 1 ) THEN

              v(:) = 0.0d0

              DO k=1,3
              v(:) = v(:) + grot(:,k,is) * r12(k)
              ENDDO

              r12(:) = v(:)
              r12e(:) = 0.0d0

              ii = nbas * ( iat - 1 )
              jj = nbas * ( jat - 1 )
              kk = 3 * ( is - 1 )

              DO j=1,nbas
              DO k=1,3
              r12e(k) = r12e(k) + gbas(kk+k,j,ir) * &
                      & ( c1 * evec(ii+j,iev) - c2 * evec(jj+j,iev) )
              ENDDO
              ENDDO

            ELSE

              ii = 3 * ( iat - 1 )
              jj = 3 * ( jat - 1 )

              DO j=1,3
              r12e(j) = c1 * evec(ii+j,iev) - c2 * evec(jj+j,iev)
              ENDDO

            ENDIF

            d = r12(1) * r12(1) + r12(2) * r12(2) + r12(3) * r12(3)
            d = DSQRT(d)

            x = r12(1) * r12e(1) + r12(2) * r12e(2) + r12(3) * r12e(3)

            r12(:) = r12(:) / d
            r12e(:) = r12e(:) / d

            DO j=1,3
            DO k=1,3
            p(j,k) = r12(j) * r12(k)
            dp(j,k) = r12e(j) * r12(k) + r12(j) * r12e(k)
            dp(j,k) = dp(j,k) - 2.0d0 * x * p(j,k)
            ENDDO
            ENDDO

            c1 = parm(1)
            c2 = parm(2) * x * d
            c3 = parm(3) * x * d

            chi(:,:) = chi(:,:) + c1 * dp(:,:) + ( c2 - c3 ) * p(:,:)

            DO j=1,3
            chi(j,j) = chi(j,j) - c2
            ENDDO

            !=== <ETA|F> ===!

            DO j=1,3
            edf(j) = edf(j) + c3 + ( c2 - c3 ) * r12(j)**2 &
                 & + 2.0d0 * c1 * r12(j) * ( -x * r12(j) + r12e(j) )
            ENDDO

          ENDDO
          ENDDO

          !=== Calculate Raman Scattering Intensity ===!

          rint = 0.0d0

          x = chi(1,1) + chi(2,2) + ch(3,3)

          rint = rlam * x * x / ( 2.0d0 * w(iev) )

          x = 0.0d0

          DO i=1,3

            x = x + chi(1,i) * chi(i,1)
            x = x + chi(2,i) * chi(i,2)
            x = x + chi(3,i) * chi(i,3)

          ENDDO

          rint = rint + rmu * x / w(iev)

          !=== OUTPUT DATA ===!

          IF ( ir == 1 ) THEN

            WRITE(3,31)w(iev),wmult,rint
!           WRITE(3,31)w(iev),wmult,rint,edf(1),edf(2),edf(3)

          ELSE

            WRITE(3,32)wmult,rint
!           WRITE(3,32)wmult,rint,edf(1),edf(2),edf(3)

          ENDIF

        ENDDO
        ENDDO

        WRITE(*,*)'VIBPROFILE is finished.'

        CLOSE(UNIT = 1)
        CLOSE(UNIT = 2)
        CLOSE(UNIT = 3)

        !=== DEALLOCATE ARRAYS ===!

        DEALLOCATE (rat,itype,prob,w,evec,lbd,lneigh,nneigh)
        IF ( isym == 1 ) DEALLOCATE (gbas,grot)

        !=== Format Statements ===!

 10     FORMAT(3E24.16)
 30     FORMAT(A,3A16)
 31     FORMAT(1X,3E16.8)
 32     FORMAT(17X,2E16.8)

! 31     FORMAT(1X,6E16.8)
! 32     FORMAT(17X,5E16.8)

      END PROGRAM VIBPROFILE

! ==============================================================================
! Subroutine: NEIGHBORS (RAT,ITYPE,LNEIGH,NNEIGH,NAT,NNMAX)
! 
! Purpose: Constructs a neighborlist of the nearest neighbors of each atoms
!          which can then be used to calculate a bond list.
!
! Method:
!
! Arguments:
!
!             RAT - (3,NAT) List of coordinates for each atom.
!           ITYPE - (NAT) List of atom types.
!          LNEIGH - (NNMAX,NAT) List Of neighbors.
!          NNEIGH - (NAT) Number of neighbors per atom.
!             NAT - Total number of atoms in the system.
!           NNMAX - Maximum number of neighbors per atom.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules -
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE NEIGHBORS (RAT,ITYPE,LNEIGH,NNEIGH,NAT,NNMAX)

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        INTEGER, INTENT(IN) :: nat,nnmax
        INTEGER, INTENT(IN) :: itype(nat)
        INTEGER, INTENT(OUT) :: lneigh(nnmax,nat),nneigh(nat)

        DOUBLE PRECISION, INTENT(IN) :: rat(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,itp,jtp,maxbd(100)
        DOUBLE PRECISION :: rcut(100,100),r12(3),x,d

        DOUBLE PRECISION, PARAMETER :: rdef = 2.50d0  !Default Bond Distance!
        DOUBLE PRECISION, PARAMETER :: rdef = 0.30d0  !Bond Distance Offset!


        !=== Set Maximum Bonds ===!

        maxbd(:) = 4

        maxbd(1) = 1
        maxbd(6) = 4
        maxbd(7) = 4
        maxbd(8) = 2
        maxbd(15) = 4
        maxbd(16) = 2

        !=== Set Cutoff Radius to Search For Bonds ===!

        rcut(:,:) = rdef

        !=== Reasonable Peptide Bond Lengths ===!

        rcut(1,1) = 0.8d0 + roff
        rcut(1,6) = 1.1d0 + roff
        rcut(1,7) = 1.0d0 + roff
        rcut(1,8) = 1.0d0 + roff
        rcut(1,16) = 1.1d0 + roff

        rcut(6,6) = 1.4d0 + roff
        rcut(6,7) = 1.4d0 + roff
        rcut(6,8) = 1.2d0 + roff
        rcut(6,16) = 1.8d0 + roff

        rcut(7,7) = 1.2d0 + roff
        rcut(7,8) = 1.2d0 + roff
        rcut(7,16) = 1.3d0 + roff

        rcut(8,8) = 1.0d0 + roff
        rcut(8,15) = 1.5d0 + roff
        rcut(8,16) = 1.3d0 + roff
        rcut(8,20) = 1.0d0 + roff

        rcut(15,15) = 2.1d0 + roff

        rcut(16,16) = 1.9d0 + roff

        DO i=1,100
        DO j=1,i
        rcut(i,j) = rcut(j,i)
        ENDDO
        ENDDO


        nneigh(:) = 0

        DO i=1,nat
        DO j=1,nat

          IF ( i == j ) CYCLE

          itp = itype(i)
          jtp = itype(j)

          x = rcut(itp,jtp)

          r12(:) = rat(:,i) - rat(:,j)

          d = r12(1) * r12(1) + r12(2) * r12(2) + r12(3) * r12(3)
          d = DSQRT(d)

          IF ( d > x ) CYCLE

          !=== Atom j is a neighbor of Atom i ===!

          nneigh(i) = nneigh(i) + 1

          !=== Check Chemistry ===!

          IF ( nneigh(i) > maxbd(itp) ) THEN

            WRITE(*,*)'ERROR: Atom #',i,'has too many neighbors.'
            WRITE(*,*)'Atom type = ',itp,'# neighbors = ',nneigh(i)
            STOP

          ENDIF

          lneigh(nneigh(i),i) = j

        ENDDO
        ENDDO

        RETURN

      END SUBROUTINE NEIGHBORS
