! Subroutine: WriteRama
! Purpose: Calculates the Ramachandran angles of a given peptide.
!          And writes them out into an output file (output file = 3)
!          For Use with pepalyze
!          File must be in (AMBER) format.
! Written: July 28th 2005
! Last Update: May 4th 2006
! Author: Eric Dykeman

! NOTES: AMBER AMINO ACID FORMAT
!        (N TERMINAL)
!        N
!        H
!        CA
!        HA
!        RESIDUE
!        C
!        O
!        (NEXT AMINO/C TERMINAL)
!
!        THE ith PHI RAMACHANDRAN ANGLE IS GIVEN BY THE DIHEDRAL
!        C_(i-1) N_i CA_i C_i
!        THE ith PSI RAMACHANDRAN ANGLE IS GIVEN BY THE DIHEDRAL
!        N_i CA_i C_i N_(i+1)
!
!        DEPENDANCIES
!        Dihdangle - Included at end
!        Cross - Included at end
!

! **********************************************************************

      Subroutine WriteRama

         Use SystemParam, Only : RAT,ITYPE,RESN,IRES,NAT,NRES

         Implicit None

      !Variable Declaration!

      !Local Variables!

         Double Precision, Dimension(:), Allocatable :: VPHI
         Integer, Dimension(:,:), Allocatable :: LDIHD

         Double Precision FACTOR
         Integer i,j,k,iE,iS,N,NRA

      !ROUTINE!

         FACTOR = 5.72957795133d1

         iS = 1

         Do While (iS .LE. NRES)

            iE = iS + 1
            Do While (IRES(iE) .GT. 0 .AND. iE .LE. NRES)
            iE = iE + 1
            EndDo
            iE = iE - 1

            NRA = 2*(iE - iS + 1)

            Allocate (LDIHD(NRA,4),VPHI(NRA))

            !Now Locate Dihedral Angles!

            !PHI(i) = C(i-1) - N(i) - CA(i) - C(i)!
            !PSI(i) = N(i) - CA(i) - C(i) - N(i+1)!

            N = 1

            Do i=iS,iE

               j = Iabs(IRES(i))

               If (i .NE. iE)Then
               k = Iabs(IRES(i+1))
               ElseIf (iE .NE. NRES)Then
               k = Iabs(IRES(iE+1))
               Else
               k = NAT
               EndIf

               !Case 1 i = iS!

               If (i .EQ. iS)Then

                  LDIHD(N,1) = j + 1
                  LDIHD(N,2) = j
                  LDIHD(N,4) = k - 2

                  LDIHD(N+1,1) = j
                  LDIHD(N+1,3) = k - 2
                  LDIHD(N+1,4) = k

                  !Wierd Case if only 1 amino acid in Chain!

                  If (i .EQ. iE .AND. k .NE. NAT)Then
                  LDIHD(N,4) = k - 3
                  LDIHD(N+1,3) = k - 3
                  LDIHD(N+1,4) = k - 1
                  EndIf

                  !Proline Exception!

                  If (RESN(i) .NE. 'PRO')Then
                  LDIHD(N,3) = j + 4
                  LDIHD(N+1,2) = j + 4
                  Else
                  LDIHD(N,3) = j + 3
                  LDIHD(N+1,2) = j + 3
                  EndIf

               EndIf

               !Case 2 iS < i < iE!

               If (i .GT. iS .AND. i .LT. iE)Then

                  LDIHD(N,1) = j - 2
                  LDIHD(N,2) = j
                  LDIHD(N,4) = k - 2

                  LDIHD(N+1,1) = j
                  LDIHD(N+1,3) = k - 2
                  LDIHD(N+1,4) = k

                  !Proline Exception!

                  If (RESN(i) .NE. 'PRO')Then
                  LDIHD(N,3) = j + 2
                  LDIHD(N+1,2) = j + 2
                  Else
                  LDIHD(N,3) = j + 1
                  LDIHD(N+1,2) = j + 1
                  EndIf

               EndIf

               !Case 3 i = iE!

               If (i .EQ. iE)Then

                  LDIHD(N,1) = j - 2
                  LDIHD(N,2) = j
                  LDIHD(N,4) = k - 2

                  LDIHD(N+1,1) = j
                  LDIHD(N+1,3) = k - 2
                  LDIHD(N+1,4) = k - 1

                  !Case if This is not the last amino!

                  If (k .NE. NAT)Then
                  LDIHD(N,4) = k - 3
                  LDIHD(N+1,3) = k - 3
                  EndIf

                  !Proline Exception!

                  If (RESN(i) .NE. 'PRO')Then
                  LDIHD(N,3) = j + 2
                  LDIHD(N+1,2) = j + 2
                  Else
                  LDIHD(N,3) = j + 1
                  LDIHD(N+1,2) = j + 1
                  EndIf

               EndIf

               N = N + 2

            EndDo

            Call DihdAngle (VPHI,RAT,LDIHD,NRA,NAT)

            Do i=1,NRA
            VPHI(i) = VPHI(i)*FACTOR
            EndDo

            Write(3,30)NRA/2,0.0d0

            Do i=1,NRA,2
            Write(3,31)VPHI(i),VPHI(i+1)
            EndDo

            iS = iE + 1

            Deallocate(LDIHD,VPHI)

         EndDo

         Return

 30      Format(I6,E16.8)
 31      Format(2F12.6)

      End Subroutine


! Subroutine: DihdAngle
! Purpose: Calculates Dihedral Angles
! Written: July 11th 2005
! Last Update: July 20th 2005
! Author: Eric Dykeman

! NOTES: VARIABLES
!           VPHI - (NRA) Vector of Ramachandran Angles
!            RAT - (NAT,3) Array containing the atomic coordinates
!          LDIHD - (NRA,4) List of Ramachandran Dihedrals
!            NRA - Number of Ramachandran Angles
!            NAT - Number of Atoms in the system
!
!        DEPENDANCIES
!        Cross
!

! **********************************************************************

      Subroutine DihdAngle (VPHI,RAT,LDIHD,NRA,NAT)

         Implicit None

      !Variable Declaration!

         Double Precision, Intent(In) :: RAT
         Double Precision, Intent(Out) :: VPHI
         Integer, Intent(In) :: LDIHD,NAT,NRA

         Dimension VPHI(NRA),LDIHD(NRA,4),RAT(NAT,3)

      !Local Variables!

         Double Precision xN1,xN2,xLN1,xLN2,DOT,CHK,CN1N2
         Double Precision R12,R32,R43
         Integer i,j

         Dimension xN1(3),xN2(3),R12(3),R32(3),R43(3),CN1N2(3)

      !ROUTINE!

         Do i=1,NRA

            Do j=1,3
            R12(j) = RAT(LDIHD(i,1),j) - RAT(LDIHD(i,2),j)
            R32(j) = RAT(LDIHD(i,3),j) - RAT(LDIHD(i,2),j)
            R43(j) = RAT(LDIHD(i,4),j) - RAT(LDIHD(i,3),j)
            EndDo

            Call Cross(R12,R32,xN1)
            Call Cross(R43,R32,xN2)
            Call Cross(xN1,xN2,CN1N2)

            CHK = 0.0d0
            DOT = 0.0d0
            xLN1 = 0.0d0
            xLN2 = 0.0d0

            Do j=1,3
            xLN1 = xLN1 + xN1(j)**2
            xLN2 = xLN2 + xN2(j)**2
            CHK = CHK + CN1N2(j)*R32(j)
            DOT = DOT + xN1(j)*xN2(j)
            EndDo

            xLN1 = Dsqrt(xLN1)
            xLN2 = Dsqrt(xLN2)
            VPHI(i) = Dacos(DOT/(xLN1*xLN2))
            If (CHK .LT. 0.0d0)Then
            VPHI(i) = -VPHI(i)
            EndIf

         EndDo

         Return

      End Subroutine


! Subroutine: Cross
! Purpose: Finds the cross product of two vectors A and B
! Written: January 8th 2004
! Last Update: April 22nd 2005
! Author: Eric Dykeman

! NOTES: VARIABLES
!        A - Array of Order three containing coordinates of Vector A
!        B - Vector B
!        C - Vector C
!        Cross will do the following operation C = A X B and Return

! **********************************************************************

      Subroutine Cross (A,B,C)

         Implicit None

      !Variable Declaration!

         Double Precision, Intent(In) :: A,B
         Double Precision, Intent(Out) :: C

         Dimension A(3),B(3),C(3)

      !ROUTINE!

         C(1) = A(2)*B(3) - A(3)*B(2)
         C(2) = A(3)*B(1) - A(1)*B(3)
         C(3) = A(1)*B(2) - A(2)*B(1)

      Return

      End Subroutine
