PROGRAM integertype
	IMPLICIT NONE

	! This program discusses the representation of the type INTEGER
	! on computers and how code can be made portable.

	! First, consider integers which assume the DEFAULT integer range
	! Usually this will be sufficient for most purposes and indeed 
	! often corresponds to the maximum range supported by the machine.
	! Now declare an integer variable which assumes the default type.

	INTEGER :: int0
	
	! Second, consider integers whose range is explicitly selected.
	! Use the intrinsic function SELECTED_INT_KIND(R)
	! where R is such that the range of integers will be 
	! at least (+-)10**R
	! This ensures portability since any computer will guarantee 
	! the above minimum range if it is supported.

	! Define the following KIND numbers, known as KIND TYPE PARAMETERS:
	INTEGER, PARAMETER :: range1=SELECTED_INT_KIND(2)
	INTEGER, PARAMETER :: range2=SELECTED_INT_KIND(4)
	INTEGER, PARAMETER :: range3=SELECTED_INT_KIND(6)
	INTEGER, PARAMETER :: range4=SELECTED_INT_KIND(9)

	! try range5=SELECTED_INT_KIND(10). What happens?

	! now declare example variables of each integer kind.
	INTEGER(KIND=range1) :: int1
	INTEGER(KIND=range2) :: int2
	INTEGER(KIND=range3) :: int3
	INTEGER(KIND=range4) :: int4
	INTEGER :: count, count_rate,count_max

!----------------------------------------------------------------------------
! 1.	! example of a default-declared integer, checking its kind using the 
	! KIND intrinsic function, and finding the largest integer of this kind.
	! Usually this will be a 32 bit representation

	int0 = 343434343
	
	PRINT*
	PRINT*, 'The default integers on this computer: '
	PRINT*
	PRINT*, 'int0 = ',int0,'  kind = ',KIND(int0),  'largest = ',HUGE(int0)
	PRINT*

!----------------------------------------------------------------------------
! 2.	! example of an integer of KIND = range1
	! this assumes an 8-bit representation

	int1 = -127_range1
	
	PRINT*
	PRINT*, 'Integers of KIND=range1: '
	PRINT*
	PRINT*, 'int1 =',int1,'  kind =',range1, '  largest =',HUGE(int1)
	PRINT*
!----------------------------------------------------------------------------
! 3.	! example of an integer of KIND = range1 = 2
	! this assumes an 16-bit representation

	int2 = 3434_range2
	
	PRINT*
	PRINT*, 'Integers of KIND=range2: '
	PRINT*
	PRINT*, 'int2 =',int2,'  kind =',range2,  '  largest =',HUGE(int2)
	PRINT*

!----------------------------------------------------------------------------
! 4.	! example of an integer of KIND = range3 = 4
	! this assumes an 32-bit representation, normally the default

	int3 = 34343434_range3
	
	PRINT*
	PRINT*, 'Integers of KIND=range3: '
	PRINT*
	PRINT*, 'int3 =',int3,'  kind =',range3,  '  largest =',HUGE(int3)
	PRINT*

!----------------------------------------------------------------------------
! 5.	! example of an integer of KIND = range4 = 4
	! this again assumes the 32-bit representation. 
	! Note that the range specified by SELECTED_INT_KIND(9) did not 
	! increase the range compared with the default. 
	! On Tower, the default uses the maximum range that is supported. 
	! Try a variable whose KIND = SELECTED_INT_KIND(20). 
	! A compile error will result giving KIND=-1; this indicates 
	! an integer range which is not supported.

	int4 = 1343434343_range4
	
	PRINT*
	PRINT*, 'Integers of KIND=range4: '
	PRINT*

	PRINT*, 'int4 =',int4,'  kind =',range4,  '  largest =',HUGE(int4)
	PRINT*

!----------------------------------------------------------------------------
! 6. Addendum.

! System clock. Finally, note the f90 intrinsic subroutine 
! SYSTEM_CLOCK(count, count_rate,count_max) which outputs  
!	count		integer set to currrent value of system clock
!	count_rate	integer set to number of processor clock counts per sec
!	count_max	integer set to maximum value that count can have; 
!			usually maximum default integer; when count =count_max
!			then count returns to 0 

	CALL SYSTEM_CLOCK(count, count_rate,count_max)

	PRINT*, 'System clock gives '
	PRINT*
	PRINT*, 'Present count is ',count
	PRINT*, 'Processor counts per second is ',count_rate
	PRINT*, 'Maximum value of count is ',count_max
	PRINT*


END PROGRAM integertype

!----------------------------------------------------------------------------
!!    Background.
!!
!!    Integers are represented exactly on a computer and integer 
!!    arithmetic is carried out exactly (but BEWARE: integer division).
!!    However, the range of values is obviously limited and you should 
!!    be aware of what the range is.
!!    On a typical 32-bit machine, one bit is reserved for the sign (+-) 
!!    and the remaining 31 can be either 0 or 1 (assuming binary) so that 
!!    numbers from 0 to (2**31 - 1) may be represented, i.e. 2147483647 
!!    or approx 2 x 10**9.
!!    This range can obviously be increased by using more bits (e.g.64)
!!    PROVIDED this is supported on the computer you are using.
!!    Then the range is +-(2**63 -1). 
!!    Older 16-bit machines could only support integers in the range 
!!    +-(2**15 - 1) i.e. 32767.

