/*
 * NoC_lib_V5.c
 *
 *  Created on: 31 Jan 2017
 *      Author: mr589
 */

#include "xparameters.h"
#include "NoC_lib_V5.h"

#define ROUTE_N_TOKEN 0x1C1
#define ROUTE_E_TOKEN 0x1C2
#define ROUTE_S_TOKEN 0x1C3
#define ROUTE_W_TOKEN 0x1C4
#define ROUTE_I_TOKEN 0x1C5
#define ROUTE_RCAP_TOKEN 0x1C6
#define ACK_TOKEN 0xAA
#define EOP_TOKEN 0x17f

//addresses for the NoC interface Registers
//As it is just a simple PLB interface, we cannot read write registers
//and likewise cannot write to read registers
#define NOC_BASE_ADDR XPAR_NOC_IF_LITE_0_BASEADDR
//read registers
#define NoC_Status_reg (Xuint32*)(NOC_BASE_ADDR + 0)
#define NoC_Read_reg (Xuint32*)(NOC_BASE_ADDR + 4)
//write registers
#define NoC_Cntrl_reg (Xuint32*)(NOC_BASE_ADDR + 8)
#define NoC_Write_reg (Xuint32*)(NOC_BASE_ADDR + 12)

void NoC_write_blocking(Xuint32 data);

//broadcast the reset signal to all nodes
void NoC_reset()
{
	*NoC_Cntrl_reg = 0x01;
	*NoC_Cntrl_reg = 0x00;
}

//writes a word (1bit control + 8 bits data) to the NoC
//Will wait until interface is ready before sending
void NoC_write_blocking(Xuint32 data)
{
	//wait for TX busy flag to be clear
	while(*NoC_Status_reg & 0x2);
	*NoC_Write_reg = data;
}

//reads a word (1bit control + 8 bits data) from the NoC
//Will wait until data is available
Xuint32 NoC_read_blocking()
{
	//wait for RX valid flag to be set
	while((*NoC_Status_reg & 0x1 ) == 0);
	return *NoC_Read_reg;
}

//writes a word (1bit control + 8 bits data) to the NoC
//returns a 1 if write was successful, 0 otherwise
Xuint32 NoC_write_non_blocking(Xuint32 data)
{
	//check if TX busy flag isn't clear
	if(*NoC_Status_reg & 0x2)
		return 0;

	*NoC_Write_reg = data;
	return 1;
}

//reads a word (1bit control + 8 bits data) from the NoC
//returns a 1 if read was succesfull and data is valid, 0 otherwise
Xuint32 NoC_read_non_blocking(Xuint32 *data)
{
	//check if RX valid flag is set
	if(*NoC_Status_reg & 0x1 )
	{
		*data = *NoC_Read_reg;
		return 1;
	}

	return 0;
}

//sinks and checks an EOP token
void NoC_Sink_EOP()
{
	Xuint32 eop = NoC_read_blocking();
	if(eop != EOP_TOKEN)
		xil_printf("V5 ERROR: SUNK DATA WAS NOT AN EOP\n");
}

//sinks and checks an ACK token and then an EOP
void NoC_Sink_ACK()
{
	Xuint32 ack = NoC_read_blocking();
	if(ack != ACK_TOKEN)
		xil_printf("V5 ERROR: SUNK DATA WAS NOT AN ACK\n");
	NoC_Sink_EOP();
}


void NoC_Write_Sys_Packet(int node, Xuint8* data, int length, int header)
{

	int node_x = node % NOC_WIDTH;
	int node_y = node / NOC_WIDTH;
	int i;
	xil_printf("Sending Message to node %d @ (%d,%d)... ", node, node_x, node_y);
	//add a south for every i
	for(i=0; i < node_y; i++)
		NoC_write_blocking(ROUTE_S_TOKEN);
	//add a east for every j
	for(i=0; i < node_x; i++)
		NoC_write_blocking(ROUTE_E_TOKEN);
	//add internal
	NoC_write_blocking(ROUTE_I_TOKEN);

	//add header if it is non zero
	if(header)
		NoC_write_blocking(header);

	//payload

	for(i=0; i< length; i++)
	{
		NoC_write_blocking(data[i]);
	}
	//add eop
	NoC_write_blocking(EOP_TOKEN);

	xil_printf("message sent\n");
}

//assign the IDs to all nodes - the nodes use this to send stuff
//back to the Microblaze so don't forget to to it at the start!
void NoC_Assign_Node_IDs()
{
	int i;
	for(i=0; i< NOC_HEIGHT * NOC_WIDTH; i++)
	{
		xil_printf("Assigning ID to node %d...\n", i);
		char index = i;
		NoC_Write_Sys_Packet(i, &index, 1, 0);
		NoC_Sink_ACK();
		xil_printf("ID done and ack'd\n");
	}
}

//Broadcast a single command (no data) to all nodes
void NoC_Broadcast_Command(Xuint32 command)
{
	int i;
	for(i=0; i< NOC_HEIGHT * NOC_WIDTH; i++)
	{
		NoC_Write_Sys_Packet(i, 0, 0, command);
	}
}


