/*
 * core.c
 *
 *  Created on: 24 Mar 2017
 *      Author: jf878
 *
 *  Contains the functions to get and set
 *  information about the core
 */

#include "core.h"
#include "NoC_lib.h"

/* Initialise the core */
void init_core(core_t * core, Xuint8 core_ID) {

	/* Set the ID of the core */
	core->node_id = core_ID;
	node_id = core_ID;

	/* Cores are initialised with no task running */
	core->currentTask = NULL;
	core->infectingTask = NULL;

	/* Set temperatures, current=0 */
	core->Tcurrent = 0;
	core->Tcritical = 200; // Give some ability to go above given range of 0-255

	core->Taccept = 100;
	core->Treject = 150;

	/* Provide the core with its resources -
	 * In a 'real' system this would depend on
	 * the hardware attached to the core and
	 * so would be generated automatically  */

	core->resources = 0; // Start with all at 0

	/* For these, lets 'randomly' choose a 1 or a zero */
	Xuint8 i;
	/* MAT 2018-02-28: Commented out because funtion does not exist!
	 * for (i = 0; i < 5; i++) {
		if (coinFlip()) {
			core->resources |= 1 << i; // Set
		}
	}
	*/

	/* If this is the node closest to the host */
	if (core_ID == 0) {
		core->resources |= HOST_ACCESS;
	}

}

/* Increase the temp if there is a task running, decrease if not */
void update_temperature(Xuint32 timeDelta) {

	timeDelta = timeDelta/SYSTICK;

	if ((core.currentTask != NULL) && (core.Tcurrent < core.Tcritical)) {
		core.Tcurrent += 2*timeDelta;
	} else if (core.Tcurrent > timeDelta) {
		core.Tcurrent -= timeDelta;
	} else {
		core.Tcurrent = 0;
	}

}

/* Get info of an infecting task from the NoC and check the infection conditions */
Xuint8 check_infection(task_t taskToCheck) {

	Xuint8 affinity;
	Xuint8 priority;

	/* Check the cores temperature first */
	if (core.Tcurrent > core.Tcritical) {
		return 0;
	}
	if (core.Tcurrent > core.Treject) {
		core.resources |= IS_HOT; // Set the IS_HOT bit
		return 0;
	}
	if ((core.Taccept < core.Tcurrent) && (core.Tcurrent < core.Treject) && (core.resources & IS_HOT)) {
		return 0;
	}

	//TODO: What happens when we accept??
	if ((core.Taccept < core.Tcurrent) && (core.Tcurrent < core.Treject) && !(core.resources & IS_HOT)) {
		// Accept - Maybe we do a coin flip here
	}
	if (core.Tcurrent < core.Taccept) {
		core.resources &= !(IS_HOT); // Clear the IS_HOT bit
		// Accept - do nothing for this one, accept all
	}

	/* Check task affinity via Hamming Distance using  */
	affinity = (taskToCheck.requirements ^ core.resources) & 0x3F; // 0x3F as we don't care about the highest two bits
	Xuint8 count;
    while (affinity != 0) {
        /* Technique by P. Wegner, May 1960 */
        count++;
        affinity &= (affinity - 1);
    }

    // If affinity is lower, the requirements and resources are a close match
    /* Now use a weighted coin flip */
    affinity = !(weighted_coinflip(affinity)); // The closer the match the fairer the coin flip

	/* Get priority and involve this with the other two results */
    priority = weighted_coinflip(taskToCheck.priority);

    /* Combine previous results */
	return weighted_coinflip(priority + affinity);

}

/* Change from one task (or none) to a new task */
Xuint8 change_task() {

	Xuint8 data[6];
	task_t newTask;

	/* If there is no infecting task then request info from the host of the next task */
	if (core.infectingTask == NULL) {

		xil_printf("Requesting new task \r\n");

		data[0] = NEXT_TASK_REQUEST;
		data[1] = node_id;

		// Request data for next task from Host
		NoC_Write_Sys_Packet(&data[0], 2);

		NoC_Write_Sys_Packet((Xuint8 *)&(core.currentTask->parentApplication), 4);

		// Wait for a response from the host
		NoC_Recieve_Packet_Blocking((Xuint8 *)(&newTask), TASK_SIZE);

		/* Check the new task and decide if the core should keep running */
		if (check_infection(newTask)) {
			// Set this as the next running task
			*(core.currentTask) = newTask;
			return 1;
		} else {
			xil_printf("New Task Rejected");
			core.currentTask = NULL;
			return 0;
		}



		/* If there is an infecting task then switch to this task and set the infecting task to NULL */
	} else if (core.infectingTask != NULL) {

		/* Decide whether to clear the parallelizable bit, this is needed to stop tasks spreading continuously */
		/* MAT 2018-02-28: Commented out because funtion does not exist!
		 * if (coinFlip()) {
				core.infectingTask->requirements &= !(IS_PARALLELIZABLE);
		   }
		*/
		core.currentTask = core.infectingTask;
		currentTask = infectingTask;

		/* Re-set the infecting task to NULL */
		core.infectingTask = NULL;
		return 1;
	}

	return 0;
}
