

#include "NoC_lib.h"

#define TX_BUFF_SIZE 1024
#define TX_BUFF ((volatile unsigned int*)(0xC1000000))
#define RX_BUFF ((volatile unsigned int*)(0xC1000000 + (4*TX_BUFF_SIZE)))
#define NOC_CNTRL_IF (volatile unsigned int*)(0xC0000000)
#define NOC_STATUS_IF (volatile unsigned int*)(0xC0000004)
#define NOC_TX_LEN_IF (volatile unsigned int*)(0xC0000008)
#define NOC_RX_LEN_IF (volatile unsigned int*)(0xC000000C)
#define NOC_RX_BASE_IF (volatile unsigned int*)(0xC0000010)

#define RTC_GPI_ADDR (volatile Xuint32*)0x80000020

Xuint8 node_id;


//clear the control reg and set the TX/RX buffer divide at half way
void NoC_Init()
{
	*NOC_RX_BASE_IF = 1024;
	*NOC_CNTRL_IF = 0x00;
}

//acknowledge a RX when we are done with the data in the buffer
void ACK_RX()
{
	*NOC_CNTRL_IF = 0x02;
	*NOC_CNTRL_IF = 0x00;
}

//Thanks to the ECC bit in the Virtex BRAMs, we have a 9-bit fifo buffer for
//sending/recieving with the NoC
Xuint32 NoC_Read_Blocking()
{
	while(((*NOC_STATUS_IF) & 0x01) ==0 );
	Xuint32 RX_size = *NOC_RX_LEN_IF;
	return RX_size;
}

//reads a token/word - i.e. a 9-bit chunk of data from the network
Xuint32 NoC_Read_Token_Blocking()
{
	NoC_Read_Blocking();
	Xuint32 value = RX_BUFF[0];
	ACK_RX();
	return value;
}

//reads a byte - i.e. an 8-bit chunk of data from the network
Xuint8 NoC_Read_Byte_Blocking()
{
	NoC_Read_Blocking();
	Xuint32 value = RX_BUFF[0];
	ACK_RX();
	return (Xuint8)(value & 0xFF);
}


//reads an int - i.e. an 32-bit chunk of data from the network and treats it as an int
Xuint32 NoC_Read_Int_Blocking()
{
	NoC_Read_Blocking();
	Xuint32 value;
	//NoC_Dump_RX_Buff(6);
	unpack_int(&value,&RX_BUFF[0]);

	ACK_RX();
	return value;
}

//reads a token in a non blocking way.
//Returns 0 if there is no data available, or 1 if there is a token via the data pointer
Xuint32 NoC_Read_Token_Non_Blocking(Xuint32 *data)
{
	if(((*NOC_STATUS_IF) & 0x01) ==0 )
		return 0;

	Xuint32 value = RX_BUFF[0];
	ACK_RX();
	*data = value;
	return 1;
}

Xuint32 NoC_Recieve_Packet_Blocking(Xint8 *data, int max_packet_length)
{
	//wait for data to be received...
	NoC_Read_Blocking();
	//print out some info
	xil_printf("Node packet received at %d from %d, header %X\n", node_id, RX_BUFF[1],  RX_BUFF[0]);
}


//sends a packet to the host microblaze node
void NoC_Write_Sys_Packet(int* data, int length)
{
	int node_x = node_id % NOC_WIDTH;
	int node_y = node_id / NOC_WIDTH;
	int i;
	int buff_index = 0;
	//wait for TX to be free
	while(*NOC_STATUS_IF & 0x2);
	//add route back
	//add a west for every j
	for(i=0; i<node_x; i++)
	{
		//Write_NoC_Blocking(0x1C4);
		TX_BUFF[buff_index] = 0x1C4;
		buff_index++;
	}

	//add a north for every i
	for(i=0; i<node_y; i++)
	{
		TX_BUFF[buff_index] = 0x1C1;
		buff_index++;
	}

	//add a North to reach the MB from node (0,0)
	TX_BUFF[buff_index] = 0x1C1;
	buff_index++;

	//now send the payload
	for(i=0; i< length; i++)
	{
		TX_BUFF[buff_index] = data[i];
		buff_index++;
	}
	//add eop
	TX_BUFF[buff_index] = 0x17F;
	//write to length reg to start the transaction
	*NOC_TX_LEN_IF = buff_index + 1;

}

//sends a packet to a neighbouring node
//TODO: header/packet etc
void NoC_Write_Node_Packet(int node, int* data, int length, Xuint32 header)
{
	int node_x = node_id % NOC_WIDTH; //TODO: a lookup table is probably worth doing here
	int node_y = node_id / NOC_WIDTH; //TODO: Divide is even worse than modulus....
	int destn_node_x = node % NOC_WIDTH; //TODO: a lookup table is probably worth doing here
	int destn_node_y = node / NOC_WIDTH; //TODO: Divide is even worse than modulus....
	int delta_x = node_x - destn_node_x;
	int delta_y = node_y - destn_node_y;

	int i;
	int buff_index = 0;
	//wait for TX to be free
	while(*NOC_STATUS_IF & 0x2);
	//add route back
	//add a west for every -ve delta
	if(delta_x < 0)
	{
		for(i=0; i>delta_x; i--)
		{
			TX_BUFF[buff_index] = 0x1C4;
			buff_index++;
		}
	}
	else
	{
		//add a east for every +ve delta
		for(i=0; i<delta_x; i++)
		{
			TX_BUFF[buff_index] = 0x1C2;
			buff_index++;
		}
	}

	//add a south for every -ve delta
	if(delta_y < 0)
	{
		for(i=0; i>delta_y; i--)
		{
			TX_BUFF[buff_index] = 0x1C3;
			buff_index++;
		}
	}
	else
	{
		//add a north for every +ve delta
		for(i=0; i<delta_y; i++)
		{
			TX_BUFF[buff_index] = 0x1C1;
			buff_index++;
		}
	}

	//add an internal to reach the node
	TX_BUFF[buff_index] = 0x1C5;
	buff_index++;

	//if(header)
	//{
		TX_BUFF[buff_index] = header;
		buff_index++;
	//}

	//send the node we have sent from?
	TX_BUFF[buff_index] = node_id;
	buff_index++;

	//now send the payload
	for(i=0; i< length; i++)
	{
		TX_BUFF[buff_index] = data[i];
		buff_index++;
	}
	//add eop
	TX_BUFF[buff_index] = 0x17F;
	//write to length reg to start the transaction
	*NOC_TX_LEN_IF = buff_index + 1;

}


void NoC_Send_ACK()
{
	int ack = 0xAA;
	NoC_Write_Sys_Packet(&ack, 1);
}


void pack_int(int* buff, unsigned int data)
{
	buff[0] = (data >> 24) & 0xFF;
	buff[1] = (data >> 16) & 0xFF;
	buff[2] = (data >> 8) & 0xFF;
	buff[3] = (data >> 0) & 0xFF;
}

void unpack_int(unsigned int* data, int* buff)
{
	*data = buff[3];
	*data |= (buff[2] << 8);
	*data |= (buff[1] << 16);
	*data |= (buff[0] << 24);
}

void NoC_Dump_RX_Buff(int num_words)
{
	int i;
	for(i=0; i<num_words;i++)
		xil_printf("RX_BUFF %d: %x\n", i, RX_BUFF[i]);
}

Xuint32 Read_RTC()
{
	return *RTC_GPI_ADDR;
}


