/* SZTOA implementation - size_t to alphanumeric string.
 * Oliver Dixon <od641@york.ac.uk>, 2022. Public Domain. */

/* The behaviour of these functions may be modified by the following
 * compile-time definitions:
 *
 *  - _SZTOA_PAD:         always fill the provided buffer, potentially padding
 *                        the unused bytes with the first character in the
 *                        relevant alphabet.
 *
 *  - _SZTOA_TRUST_ALPHS: do not call strlen(3) to ensure a custom alphabet is
 *                        of an appropriate size to accommodate the string
 *                        representation of the integer. If this is set, it
 *                        becomes the caller's responsibility to ensure that
 *                        given alphabets are suitably large. */

#include <assert.h>
#include <errno.h>
#include <stddef.h> /* size_t */

#ifndef _SZTOA_TRUST_ALPHS
#include <string.h> /* strlen */
#endif  /* _SZTOA_TRUST_ALPHS */

/* swap: swap a couple of characters, referenced by c1 and c2. */

inline static void swap ( char * c1, char * c2 )
{
        char tmp = *c1;

        *c1 = *c2;
        *c2 = tmp;
}

/* write_buf: writes the string representation of `val` to `buffer` in the
 * appropriate `base`. If the buffer is not large enough to accommodate the
 * string, the last `sz-1` bytes of the `val` string representation will be
 * stored, and errno is set to EOVERFLOW. This function returns the number of
 * bytes written to `buffer`. */

static size_t write_buf ( char * buffer, size_t val, size_t sz,
                unsigned int base, const char * alph )
{
        unsigned char digit;
        size_t i = 0;

        do {
                buffer [ i ] = alph [ ( digit = val % base ) ];
                val /= base;

                if ( i++ == sz ) {
                        errno = EOVERFLOW;
                        return i - 1;
                }
        } while ( val );

#ifdef _SZTOA_PAD
        for ( ; i < sz; i++ )
                buffer [ i ] = *alph;
#endif /* _SZTOA_PAD */

        return i;
}

/* sztoa: convert size_t `val` to its string representation of an arbitrary
 * `base`, storing the result in the provided `buffer`, of size `sz`. For bases
 * larger than hexadecimal, a NULL-terminated custom alphabet must be provided
 * through `prov_alph`. If `base` is not of an appropriate value, errno is set
 * to EINVAL. This function returns the buffer address. The `buffer` is
 * NULL-terminated, unless errno is set to EINVAL, in which case it is
 * unmodified. */

char * sztoa ( char * buffer, size_t val, size_t sz, unsigned int base,
                const char * prov_alph )
{
        static const char def_alph [ ] = "0123456789ABCDEF";
        static const unsigned int def_alph_len =
                sizeof ( def_alph ) / sizeof ( *def_alph );

        /* Use the default alphabet unless a suitable one is provided. */
#ifdef _SZTOA_TRUST_ALPHS
        /* Highly iterative calls passing the same alphabet may obviate the
         * strlen(3) call, if significantly beneficial to performance. */
        const char * alph = ( prov_alph != NULL ) ? prov_alph : def_alph;
#else  /* _SZTOA_TRUST_ALPHS */
        const char * alph = ( prov_alph != NULL && base
                <= strlen ( prov_alph ) ) ? prov_alph : def_alph;
#endif /* _SZTOA_TRUST_ALPHS */

        if ( ( alph == def_alph && base >= def_alph_len ) || base < 2 ) {
                /* Base is too large for default alphabet, or the base is
                 * invalid. */
                errno = EINVAL;
                return buffer;
        }

        if ( sz < 2 || buffer == NULL ) {
                errno = EOVERFLOW; /* The buffer would always overflow. */
                if ( sz == 1 )
                        *buffer = '\0';

                return buffer;
        }

        val = ( sz = write_buf ( buffer, val, sz - 1, base, alph ) ) >> 1;

        /* Reverse the buffer contents. */
        for ( size_t i = 0; i < val; i++ )
                swap ( & ( buffer [ i ] ), & ( buffer [ sz - i - 1 ] ) );

        buffer [ sz ] = '\0';
        return buffer;
}

