#!/usr/bin/env python
#UCAS molecular simulation, Summer 2016
#Written by Ryan Cocking & Robert Forrest
import numpy as npy
from sys import stdout, path
from time import sleep
import os
import math
import time
import thread
from Queue import Queue

import itp_visuals as vis
import itp_physics as phys
import itp_params  as params


from cProfile import Profile
from pstats import Stats
prof = Profile()
params.basedir = path[0][0:-7] # Find base directory (remove (hopefully) /source)

def abort(ErrMsg):
    print "Error :", ErrMsg
    os._exit(1)
    

def run(scenario):
    
    #Declare global variables which can be accessed by the GUI
    global paused, onestep, restart, end, fresh_run, continuing, another_frame
    global current_T, T_average, pressure, t
    global picked_atom_index
    global displacement
    global dragging
    global scenario_out

    scenario_out = scenario


    #This is the first time the sim is running
    fresh_run           = True

    end                 = False
    
    while end == False:
        #Should the sim stop?
        #----------------------------------------------------------------------------------------------------#
        #                                           MAIN PROGRAM                                             #
        #----------------------------------------------------------------------------------------------------#


        #----------------------------#
        #           UNITS            #
        #----------------------------#

         # Get scenario and parameters
        filename        = params.basedir+'/source/scenarios/'+scenario+'.txt'
        params.read_params(filename)

        t                       = 0.0                                   # Current time

        # GUI controls
        paused                  = True
        onestep                 = False
        restart                 = False
        end                     = False
        fps                     = 0                                     # Frames per second

        T_running_total         = [0.0 for i in xrange(params.nav)]            # Running average for temperature
        T_average               = 0.0                                   # Average temperature over several timesteps

        # Energy
        KE                      = 0.0                                   # Kinetic Energy of current step
        PE                      = 0.0                                   # Potential Energy of current step
        KE_tot                  = 0.0                                   # Total kinetic energy of the system
        PE_tot                  = 0.0                                   # Total potential energy of the system

        picked_atom_index       = -1                                    # Information from vis about atom being dragged
        displacement            = npy.zeros(2)
        dragging                = False


        #----------------------------#
        #         INITIALISE         #
        #----------------------------#
        #Spawn atoms, get initial positions and velocities
        if (params.AtomFile != ''):
            # Read file
            params.N, pos = loadXYZ(params.basedir+'/source/scenarios/'+params.AtomFile)
            params.box_width = npy.ceil(params.size + (params.N/10.))
            params.ndof = params.Ndim*params.N
        else:
            # Either take initial pos and vel from scenario file or generate them.
            params.box_width               = npy.ceil(params.size + (params.N/10.))
            if ('pos' in params.raw_params):
                try: 
                    pos                 = npy.asarray(eval(params.raw_params['pos']))
                except KeyError:
                    abort('No atoms specified in '+scenario)
                if (pos.shape != (params.N,params.Ndim)):
                    abort('Atom number or dimensions different in pos')
            else:
                pos = phys.spawn(params.lattice_constant, params.box_width, params.defect)

        
        # Atoms which are fixed in place
        fixed = params.raw_params.get('fixed','')
        if (fixed != ''):
            fixed = map(int,fixed.split(','))
            params.ndof  -= params.Ndim*len(fixed)
        # Define velocities by: VelFile, vel, v, init_T
        if params.VelFile != '':
            try_N,vel = loadXYZ(params.basedir+'/source/scenarios/'+params.VelFile)
            if try_N != params.N:
                abort('Number of velocities not equal to number of atoms')
        elif 'vel' in params.raw_params:
            vel = npy.asarray(eval(params.raw_params['vel']))
        elif 'v' in params.raw_params:
            vel = npy.asarray(eval(params.raw_params['v']))
        else:
            vel = phys.initialise_temp()
        if (vel.shape != (params.N,params.Ndim)):
            abort('Atom number or dimensions different in vel')
        
        # Create bonds between all atoms, or take bond information from scenario file
        bonds                   = eval(params.raw_params.get('bonds','[]'))
        if(bonds is True):
            bonds = phys.create_bonds(pos)
        elif not isinstance(bonds, (list,tuple,npy.ndarray)):
            abort('Bonds not set correctly')
            
        # Calculate initial forces between atoms
        force, PE, pressure     = phys.compute_forces(pos, vel,picked_atom_index, displacement, dragging, bonds, fixed)

        #Calculate the kinetic energies of all the atoms
        KE = 0.5*reduce(lambda ke, v: ke + npy.dot(v,v), vel, 0)

        #Calculate total KE and PE
        KE_tot                  = KE
        PE_tot                  = PE - params.potential_correction

        #Calculate the total energy of the system
        E_tot                   = KE_tot + PE_tot

        #Calculate the temperature of the system
        current_T               = phys.calculate_temperature(KE)

        #Create the visuals, if they haven't been already created
        if fresh_run == True:
            # VPython initially calls the window VPython
            if(vis.scene.title == "VPython"):
                vis.build_visuals()
            else:
                vis.destroy_objects()
                vis.update_scene()
                
            vis.build_atoms(pos)
            vis.bonds           = bonds
            vis.fixed           = fixed
            q = Queue()
            if(params.Ndim == 2):
                q2 = Queue()
                thread.start_new_thread(vis.update_visuals, (q,q2,))
            else:
                thread.start_new_thread(vis.update_visuals, (q,))
            fresh_run   = False

        #----------------------------#
        #             RUN            #
        #----------------------------#

        k               = 0
        average_counter = 1
        another_frame   = True
        while another_frame:
            # Update dt based on the current temperature, preventing crashes
            if (not paused):
                paused = safety_check(current_T, paused)
                # params.dt, paused = update_dt(current_T, dt, paused)
                
            #Sleep the thread while paused
            while paused == True:
                # Break out of loop, only to come back to be paused again, thus incrementing by small amount of time
                if onestep == True:
                    onestep             = False
                    break
                sleep(0.05)

            # Break out of simulation loop, go back to initialisation
            if restart == True:
                paused = False
                onestep = False
                break            

            # Initialise energy totals for this step
            KE_tot              = 0
            PE_tot              = 0

            # Perform several physics calculations per visual step
            for i in xrange(params.visual_skip):

                # Calculate positions and velocities
                pos, vel_temp           = phys.vv_1(pos, vel, force)

                # Calculate forces and potential energies
                force, PE, pressure     = phys.compute_forces(pos, vel_temp, picked_atom_index, displacement, dragging, bonds, fixed)

                # Calculate new velocities and kinetic energies
                vel, KE                 = phys.vv_2(vel_temp, force)
                
                # Sum total energies
                KE_tot                  += KE
                PE_tot                  += PE

                t                       += params.dt

            # Get the correct energy totals
            KE_tot              = KE_tot / params.visual_skip
            PE_tot              = PE_tot / params.visual_skip
            
            #Update potential energy with the energy correction
            PE_tot              = PE_tot - params.potential_correction

            # Calculate total energy
            E_tot               = PE_tot + KE_tot

            # Calculate the total temperature of the system
            current_T           = phys.calculate_temperature(KE)

            # Calculate average temperature over recent timesteps
            T_running_total[k%params.nav] = current_T
            T_average = (round(sum(T_running_total)/params.nav,2))
            k += 1

            # Update the visuals with new data
            q.put(pos)
            q.put(vel)
            q.join()

            #Get dragging data from vis thread
            if(params.Ndim == 2):
                dragging = q2.get()
                q2.task_done()
                
                if(dragging):
                    picked_atom_index       = q2.get()
                    q2.task_done()
                    displacement            = q2.get()
                    q2.task_done()


def update_dt(T,paused):

    if (T > 1000):
        paused = True
        print "Simulation too hot, pausing"
    else:
        if (T > 1/(params.safety_factor*params.dt**2)):
            dt = abs(npy.max(npy.sqrt(1/(params.safety_factor*T)) - 0.001, 0.001))
    return dt, paused

def safety_check(T, paused):
    if (T > 1000):
        paused = True
        print "Simulation too hot, pausing"
    else:
        if (T > 1/(params.safety_factor*params.dt**2)):
            paused = True
            print "dt too large? system going unstable, recommend:", round(abs(npy.max(npy.sqrt(1./(params.safety_factor*T)) - 0.001, 0.001)),3)
    return paused

def loadXYZ(filename):
    import re
    global N
    
    out_data = []
    with open(filename) as f_in:
        for line in params.nonblank_lines(f_in):
            words = line.split()
            if (re.match('[0-9]+\s*$',line)):
                N = int(words[0])
            elif (re.match('\s*[A-Z][a-z]{,2}(\s+[0-9.e+-]+){3}$',line)):
                out_data.append(npy.asarray(words[1:params.Ndim+1],dtype=float))
            else:
                continue
    if (len(out_data) != N):
        abort('Number of atoms does not match N in '+filename)
    out_data = npy.asfortranarray(out_data)
    return N,out_data
